#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using System.Threading;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTest.VAL;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
#endregion

namespace gov.va.med.vbecs.BOL.UnitTests
{

	
	

	[TestFixture]
	public class BOL_PatientUnitIssue : BaseTest 
	{
		#region Variables

		private BOL.PatientUnitIssue.IssuedUnits _issuedUnits;
		private string _user;

		private BOL.PatientUnitIssue _patientUnitIssue;
		// This is the specimen that is created first and not the one associated
		// with the PatientUnitSelecion object that we're testing with.
		Guid _specimenOneGuid = Guid.Empty;
		// This is the specimen that is created second and the one associated
		// with the PatientUnitSelecion object that we're testing with.
		Guid _specimenTwoGuid = Guid.Empty;

		// OrderedTest associated with the specimen tests that are associated with
		// the specimen that is associated with the PUS object.  Got that?
		Guid _orderedTestGuid = Guid.Empty;

		// ToDo: Not sure if I need this one or not...
		//BOL.OrderedComponent _orderedComponent;
		Guid _patientGuid;

		Guid _orderedComponentGuid;

		// Ordered unit associated with this PUI
		Guid _orderedUnitGuid;

		// Specime Test guid associated with Rh tests
		Guid _specimenTestRhTestOneGuid;
		Guid _specimenTestRhTestTwoGuid;

		private string _cleanupSQL;
		private string _cleanupSQLTwo;

        private const string IRR = "Irradiated";
        private const string LEU = "Leukoreduced";
        private const string WASRBC = "Washed RBC";
        private const string WASPLT = "Washed Platelet";
        private const string CMV = "CMV Negative";
        private const string SC = "SC Negative";

		#endregion

		[SetUp]
		protected void SetUp()
		{
			this.BaseSetUp();

			//ToDo: Uncomment
			//VistALinkUnitTestHarness.RequireConnection();

			if (this.RefreshData)
			{
				_patientGuid = CreatePatient();
				//_patient = new BOL.Patient( (Guid) dtRow[TABLES.Patient.PatientGuid]);
				this.DeletePatientData(_patientGuid);
				_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			}

			_issuedUnits = new gov.va.med.vbecs.BOL.PatientUnitIssue.IssuedUnits(System.DateTime.Today);
		}

		[TearDown]
		protected void TearDown()
		{
			try
			{
				VistALinkUnitTestHarness.ReleaseConnection();
				if (_cleanupSQL != null && _cleanupSQL.Trim().Length > 0) UnitTestUtil.RunSQL(_cleanupSQL);
			}
			catch{}
		}


		#region inner classes

		[Test]
		public void IssueMessagesEventArgs_IssueMessagesEventArgs_Pass()
		{
			Hashtable issueMessages = new Hashtable();
			issueMessages.Add( "Test", "Test" );
			//
			DataRow dr = UnitTestUtil.RunSQLGetDataRow( "SELECT TOP 1 BloodUnitGuid FROM IssuedUnit" );
			Guid bloodUnitGuid = (Guid)dr[TABLES.IssuedUnit.BloodUnitGuid];
			BOL.BloodUnit bloodUnit = new BloodUnit( bloodUnitGuid );
			//
			BOL.PatientUnitIssue.IssueMessagesEventArgs args = 
				new BOL.PatientUnitIssue.IssueMessagesEventArgs( issueMessages, bloodUnit );
			//
			Assert.IsNotNull( args );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssueMessagesEventArgs_IssueMessagesEventArgs_Fail()
		{
		}

		[Test]
		public void IssueMessagesEventArgs_Messages_Pass()
		{
			Hashtable issueMessages = new Hashtable();
			issueMessages.Add( "Test", "Test" );
			//
			DataRow dr = UnitTestUtil.RunSQLGetDataRow( "SELECT TOP 1 BloodUnitGuid FROM IssuedUnit" );
			Guid bloodUnitGuid = (Guid)dr[TABLES.IssuedUnit.BloodUnitGuid];
			BOL.BloodUnit bloodUnit = new BloodUnit( bloodUnitGuid );
			//
			BOL.PatientUnitIssue.IssueMessagesEventArgs args = 
				new BOL.PatientUnitIssue.IssueMessagesEventArgs( issueMessages, bloodUnit );
			//
			Assert.IsNotNull( args.Messages.Equals( issueMessages ) );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void IssueMessagesEventArgs_Messages_Fail()
		{
		}

		[Test]
		public void IssueMessagesEventArgs_BloodUnit_Pass()
		{
			Hashtable issueMessages = new Hashtable();
			issueMessages.Add( "Test", "Test" );
			//
			DataRow dr = UnitTestUtil.RunSQLGetDataRow( "SELECT TOP 1 BloodUnitGuid FROM IssuedUnit" );
			Guid bloodUnitGuid = (Guid)dr[TABLES.IssuedUnit.BloodUnitGuid];
			BOL.BloodUnit bloodUnit = new BloodUnit( bloodUnitGuid );
			//
			BOL.PatientUnitIssue.IssueMessagesEventArgs args = 
				new BOL.PatientUnitIssue.IssueMessagesEventArgs( issueMessages, bloodUnit );
			//
			Assert.IsNotNull( args.BloodUnit.Equals( bloodUnit ) );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void IssueMessagesEventArgs_BloodUnit_Fail()
		{
		}

		#endregion


		#region constructors

		[Test]
		public void Ctor_Pass()
		{
			Assert.IsTrue(_patientUnitIssue.BrokenRulesCount==0);
		}
		[Test]
		public void Ctor_Fail()
		{
			// Check to make sure correct number of broken rules are being triggered
			_patientUnitIssue.SetInitialBrokenRules("DlgIssueBloodDetails", "default");
			Assert.IsTrue(_patientUnitIssue.BrokenRulesCount==2, "Testing broken rules agains DlgIssueBloodDetails form");

			_patientUnitIssue.SetInitialBrokenRules("FrmIssueBloodComponents", "default");
			Assert.IsTrue(_patientUnitIssue.BrokenRulesCount==1, "Testing broken rules agains FrmIssueBloodComponents form");
		}

		#endregion


		#region properties

		[Test]
		public void Patient_Pass()
		{
			// Patient should never be null
			Assert.IsTrue(_patientUnitIssue.Patient != null && 
				_patientUnitIssue.Patient.VistaPatientId == new BOL.Patient(_patientGuid).VistaPatientId);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Patient_Fail()
		{
		}


		[Test]
		public void OrderingPhysician_Pass()
		{
			string foo = "foo";
			_patientUnitIssue.OrderingPhysician = foo;
			Assert.IsTrue(_patientUnitIssue.OrderingPhysician == foo);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void OrderingPhysician_Fail()
		{
		}


		[Test]
		public void IssueDate_Pass()
		{
			_patientUnitIssue.SetInitialBrokenRules("DlgIssueBloodDetails", "default");
			int brokenRulesBefore = _patientUnitIssue.BrokenRulesCount;

			// Valid
			_patientUnitIssue.IssueDate =  VBECSDateTime.GetDivisionCurrentDateTime();
			Assert.IsTrue(brokenRulesBefore == _patientUnitIssue.BrokenRulesCount &&
				_patientUnitIssue.IssueDate != DateTime.MinValue && _patientUnitIssue.IssueDate != DateTime.MaxValue, "Issue date set to now");
			
			_patientUnitIssue.IssueDate = DateTime.Now.AddDays(-1);
			Assert.IsTrue(brokenRulesBefore == _patientUnitIssue.BrokenRulesCount, "issue date set to yesterday");
		}
		[Test]
		public void IssueDate_Fail()
		{
			_patientUnitIssue.SetInitialBrokenRules("DlgIssueBloodDetails", "default");
			int brokenRulesBefore = _patientUnitIssue.BrokenRulesCount;

			// Invalid
			_patientUnitIssue.IssueDate = DateTime.Now.AddDays(1);
			Assert.IsTrue( (brokenRulesBefore + 1) == _patientUnitIssue.BrokenRulesCount);
		}


		[Test]
		public void IssuedBy_Pass()
		{
			_patientUnitIssue.SetInitialBrokenRules("DlgIssueBloodDetails", "default");
			int brokenRulesBefore = _patientUnitIssue.BrokenRulesCount;

			// not broken by default
			_patientUnitIssue.IssuedBy = GetValidUser();
			Assert.IsTrue( (brokenRulesBefore) == _patientUnitIssue.BrokenRulesCount && _patientUnitIssue.IssuedBy.Length > 0);
		}
		[Test]
		public void IssuedBy_Fail_EmptyString()
		{
			_patientUnitIssue.SetInitialBrokenRules("DlgIssueBloodDetails", "default");
			int brokenRulesBefore = _patientUnitIssue.BrokenRulesCount;

			_patientUnitIssue.IssuedBy = String.Empty;
			Assert.IsTrue( (brokenRulesBefore+1) == _patientUnitIssue.BrokenRulesCount);
		}
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void IssuedBy_Fail_NullString()
		{
			_patientUnitIssue.SetInitialBrokenRules("DlgIssueBloodDetails", "default");
			int brokenRulesBefore = _patientUnitIssue.BrokenRulesCount;

			_patientUnitIssue.IssuedBy = null;
			Assert.IsTrue( (brokenRulesBefore+1) == _patientUnitIssue.BrokenRulesCount);
		}


		[Test]
		public void AllUnitsInvalidMessage_Pass()
		{
			// By default, there will be no valid units associated with this PUI
			Assert.IsNotNull(_patientUnitIssue.AllUnitsInvalidMessage);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void AllUnitsInvalidMessage_Fail()
		{
		}


		[Test]
		public void Transporter_Pass()
		{
			_patientUnitIssue.SetInitialBrokenRules("DlgIssueBloodDetails", "default");
			int brokenRulesBefore = _patientUnitIssue.BrokenRulesCount;

			// not broken by default
			_patientUnitIssue.Transporter = "Foo";
			string transporterValue = _patientUnitIssue.Transporter; 
			Assert.IsTrue( (brokenRulesBefore-1) == _patientUnitIssue.BrokenRulesCount && transporterValue == "Foo");
		}
		[Test]
		public void Transporter_Fail_NullString()
		{
			_patientUnitIssue.SetInitialBrokenRules("DlgIssueBloodDetails", "default");
			int brokenRulesBefore = _patientUnitIssue.BrokenRulesCount;

			// not broken by default
			_patientUnitIssue.Transporter = null;
			Assert.IsTrue( (brokenRulesBefore) == _patientUnitIssue.BrokenRulesCount);
		}
		[Test]
		public void Transporter_Fail_EmptyString()
		{
			_patientUnitIssue.SetInitialBrokenRules("DlgIssueBloodDetails", "default");
			int brokenRulesBefore = _patientUnitIssue.BrokenRulesCount;

			// not broken by default
			_patientUnitIssue.Transporter = string.Empty;
			Assert.IsTrue( (brokenRulesBefore) == _patientUnitIssue.BrokenRulesCount);
		}


		[Test]
		public void IssueToLocation_Pass()
		{
			_patientUnitIssue.SetInitialBrokenRules("DlgIssueBloodDetails", "default");
			ArrayList locations = _patientUnitIssue.GetLocations();
			int brokenRulesBefore = _patientUnitIssue.BrokenRulesCount;

			// not broken by default
			_patientUnitIssue.IssueToLocation = (string) locations[0];
			string issueToLocationValue = _patientUnitIssue.IssueToLocation;
			Assert.IsTrue( (brokenRulesBefore-1) == _patientUnitIssue.BrokenRulesCount && issueToLocationValue == _patientUnitIssue.IssueToLocation);

			_patientUnitIssue.IssueToLocation = (string) locations[0] + "'";
			// find a location with a quote:
			for (int i=0; i<locations.Count; i++)
			{
				if (locations.IndexOf("'") != -1)
				{
					_patientUnitIssue.IssueToLocation = (string) locations[i];
					Assert.IsTrue(_patientUnitIssue.IssueToLocation == (string) locations[i]);
					break;
				}
			}
			
		}
		[Test]
		public void IssueToLocation_Fail_EmptyString()
		{
			_patientUnitIssue.SetInitialBrokenRules("DlgIssueBloodDetails", "default");
			_patientUnitIssue.GetLocations();
			int brokenRulesBefore = _patientUnitIssue.BrokenRulesCount;

			// not broken by default
			_patientUnitIssue.IssueToLocation = string.Empty;
			Assert.IsTrue( (brokenRulesBefore) == _patientUnitIssue.BrokenRulesCount);
		}


		[Test]
		public void RemoteStorageIndicator_Pass()
		{
			_patientUnitIssue.SetInitialBrokenRules("DlgIssueBloodDetails", "default");
			int brokenRulesBefore = _patientUnitIssue.BrokenRulesCount;

			// not broken by default
			_patientUnitIssue.RemoteStorageIndicator = false;
			Assert.IsTrue( (brokenRulesBefore) == _patientUnitIssue.BrokenRulesCount);
		}
		[Test]
		public void RemoteStorageIndicator_Fail()
		{
			_patientUnitIssue.SetInitialBrokenRules("DlgIssueBloodDetails", "default");
			int brokenRulesBefore = _patientUnitIssue.BrokenRulesCount;

			// not broken by default
			_patientUnitIssue.RemoteStorageIndicator = true;
			Assert.IsTrue( (brokenRulesBefore+1) == _patientUnitIssue.BrokenRulesCount 
				&& _patientUnitIssue.RemoteStorageIndicator == true);
		}


		[Test]
		public void RemoteStorageIdentity_Pass()
		{
			_patientUnitIssue.SetInitialBrokenRules("DlgIssueBloodDetails", "default");
			int brokenRulesBefore = _patientUnitIssue.BrokenRulesCount;

			// not broken by default
			_patientUnitIssue.RemoteStorageIndicator = true;
			_patientUnitIssue.RemoteStorageIdentity = "Foo";
			Assert.IsTrue( (brokenRulesBefore) == _patientUnitIssue.BrokenRulesCount 
				&& _patientUnitIssue.RemoteStorageIdentity == "Foo");
		}
		[Test]
		public void RemoteStorageIdentity_Fail_NullString()
		{
			_patientUnitIssue.SetInitialBrokenRules("DlgIssueBloodDetails", "default");
			int brokenRulesBefore = _patientUnitIssue.BrokenRulesCount;

			// not broken by default
			_patientUnitIssue.RemoteStorageIndicator = true;
			_patientUnitIssue.RemoteStorageIdentity = null;
			Assert.IsTrue( (brokenRulesBefore+1) == _patientUnitIssue.BrokenRulesCount);
		}
		[Test]
		public void RemoteStorageIdentity_Fail_EmptyString()
		{
			_patientUnitIssue.SetInitialBrokenRules("DlgIssueBloodDetails", "default");
			int brokenRulesBefore = _patientUnitIssue.BrokenRulesCount;

			// not broken by default
			_patientUnitIssue.RemoteStorageIndicator = true;
			_patientUnitIssue.RemoteStorageIdentity = string.Empty;
			Assert.IsTrue( (brokenRulesBefore+1) == _patientUnitIssue.BrokenRulesCount);
		}


		[Test]
		public void Units_Pass()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue( _patientUnitIssue.Units != null && _patientUnitIssue.Units.Count == 1 );
		}
		[Test]
		public void Units_Fail()
		{
			Assert.IsTrue( _patientUnitIssue.Units != null && _patientUnitIssue.Units.Count == 0 );
		}


		[Test]
		public void EmergencyUnits_Pass()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.O, Common.RH.Positive, Common.ComponentClass.RBC);
			BOL.BloodUnit bloodUnitTwo = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.O, Common.RH.Positive, Common.ComponentClass.RBC);
			Guid [] bloodUnitGuids = {bloodUnit.BloodUnitGuid, bloodUnitTwo.BloodUnitGuid};
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", true, bloodUnitGuids);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue( _patientUnitIssue.EmergencyUnits != null);
		}
		[Test]
		public void EmergencyUnits_Fail()
		{
			Assert.IsTrue( _patientUnitIssue.EmergencyUnits != null && _patientUnitIssue.EmergencyUnits.Count == 0 );
		}


		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void UnitOK_Pass()
		{
			_patientUnitIssue.UnitOK = true;

		}
		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void UnitOK_Fail()
		{
		}


		[Test]
		public void IssueExceptions_Pass()
		{
			Assert.IsTrue(_patientUnitIssue.IssueExceptions.Count == 0);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void IssueExceptions_Fail()
		{
		}



		#region IssuedUnits

		[Test]
		public void IssuedUnits_IsSynchronized_Pass()
		{
			
			Assert.AreEqual(_issuedUnits.IsSynchronized, false);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssuedUnits_IsSynchronized_Fail()
		{
		}


		// ToDo: Update once data is updated
		[Test]
		public void IssuedUnits_Count_Pass()
		{
			Assert.IsTrue(_issuedUnits.Count==0);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssuedUnits_Count_Fail()
		{
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void IssuedUnits_CopyTo_Pass()
		{
			_issuedUnits.CopyTo(null, 0);
			//Assertion.AreEqual(_issuedUnits.CopyTo(), false);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssuedUnits_CopyTo_Fail()
		{
		}

		[Test]
		public void IssuedUnits_SyncRoot_Pass()
		{
			
			Assert.AreEqual(_issuedUnits.IsSynchronized, false);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssuedUnits_SyncRoot_Fail()
		{
		}

		#endregion

		#endregion


		#region methods
        bool ValidTRMessage(string reqs, string message, bool elevated)
        {
            if (reqs.IndexOf(IRR) == -1 && message.IndexOf(IRR) != -1) return false;
            if (reqs.IndexOf(IRR) > -1 && message.IndexOf(IRR) == -1) return false;

            if (reqs.IndexOf(LEU) == -1 && message.IndexOf(LEU) != -1) return false;
            if (reqs.IndexOf(LEU) > -1 && message.IndexOf(LEU) == -1) return false;

            if (reqs.IndexOf(LEU) == -1 && message.IndexOf(LEU) != -1) return false;
            if (reqs.IndexOf(LEU) > -1 && message.IndexOf(LEU) == -1) return false;

            if (reqs.IndexOf(WASRBC) == -1 && message.IndexOf(WASRBC) != -1) return false;
            if (reqs.IndexOf(WASRBC) > -1 && message.IndexOf(WASRBC) == -1) return false;

            if (reqs.IndexOf(CMV) == -1 && message.IndexOf(CMV) != -1) return false;
            if (reqs.IndexOf(CMV) > -1 && message.IndexOf(CMV) == -1) return false;

            if (reqs.IndexOf(SC) == -1 && message.IndexOf(SC) != -1) return false;
            if (reqs.IndexOf(SC) > -1 && message.IndexOf(SC) == -1) return false;

            if (elevated && message.IndexOf("You do not have the proper security to issue this unit.") > -1) return false;
            if (!elevated && message.IndexOf("You do not have the proper security to issue this unit.") == -1) return false;

            return true;
        }



        [Test]
        public void ComponentRequirements_RBC_Fail()
        {
            ArrayList reqs = new ArrayList();
            reqs.Add(IRR);					// Test case #1
            reqs.Add(LEU);					// Test case #2
            reqs.Add(WASRBC);				// Test case #3
            reqs.Add(CMV);					// Test case #4
            reqs.Add(SC);					// Test case #5

            reqs.Add(LEU + ", " + IRR);			// Test case #6
            reqs.Add(IRR + ", " + WASRBC);			// Test case #7
            reqs.Add(CMV + ", " + IRR);			// Test case #8
            reqs.Add(SC + ", " + IRR);				// Test case #9
            reqs.Add(WASRBC + ", " + LEU);			// Test case #10
            reqs.Add(LEU + ", " + CMV);			// Test case #11
            reqs.Add(SC + ", " + LEU);				// Test case #12
            reqs.Add(WASRBC + ", " + CMV);			// Test case #13
            reqs.Add(WASRBC + ", " + SC);			// Test case #14
            reqs.Add(CMV + ", " + SC);				// Test case #15

            reqs.Add(LEU + ", " + IRR + ", " + WASRBC);	// Test case #16
            reqs.Add(LEU + ", " + CMV + ", " + IRR);		// Test case #17
            reqs.Add(SC + ", " + IRR + ", " + LEU);		// Test case #18
            reqs.Add(CMV + ", " + WASRBC + ", " + IRR);	// Test case #19
            reqs.Add(WASRBC + ", " + SC + ", " + IRR);	// Test case #20
            reqs.Add(CMV + ", " + SC + ", " + IRR);		// Test case #21
            reqs.Add(LEU + ", " + CMV + ", " + SC);		// Test case #22
            reqs.Add(SC + ", " + WASRBC + ", " + CMV);	// Test case #23
            reqs.Add(CMV + ", " + WASRBC + ", " + LEU);	// Test case #24
            reqs.Add(WASRBC + ", " + SC + ", " + LEU);	// Test case #25

            reqs.Add(WASRBC + ", " + CMV + ", " + IRR + ", " + LEU);		// Test case #26
            reqs.Add(SC + ", " + WASRBC + ", " + IRR + ", " + LEU);		// Test case #27
            reqs.Add(IRR + ", " + SC + ", " + CMV + ", " + LEU);			// Test case #28
            reqs.Add(SC + ", " + CMV + ", " + IRR + ", " + WASRBC);		// Test case #29
            reqs.Add(SC + ", " + WASRBC + ", " + LEU + ", " + CMV);		// Test case #30

            reqs.Add(WASRBC + ", " + CMV + ", " + IRR + ", " + SC + ", " + LEU);	// Test case #31



            BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.O, Common.RH.Negative, Common.ComponentClass.RBC);
            Patient patient = new Patient(_patientGuid);
            this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
            _patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));


            // No TRs here
            Hashtable results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
            Assert.IsTrue(results.Count == 0);


            // First, do the elevated security level
            for (int i = 0; i < reqs.Count; i++)
            {
                this.SetupCRs(
                    patient,
                    reqs[i].ToString().IndexOf(IRR) != -1,
                    reqs[i].ToString().IndexOf(LEU) != -1,
                    reqs[i].ToString().IndexOf(WASRBC) != -1,
                    reqs[i].ToString().IndexOf(WASPLT) != -1,
                    reqs[i].ToString().IndexOf(CMV) != -1,
                    reqs[i].ToString().IndexOf(SC) != -1
                );


                _patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist)); // need to recreate the object to account for caching
                results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
                Assert.IsTrue(results.Count == 1);

                IDictionaryEnumerator enumerator = results.GetEnumerator();
                enumerator.MoveNext();

                Common.StrRes.StringResourceItem message = (Common.StrRes.StringResourceItem)enumerator.Key;

                Assert.IsTrue(ValidTRMessage(reqs[i].ToString(), message.ResString, true));
            }

            // Next, do the lower security level
            for (int i = 0; i < reqs.Count; i++)
            {
                this.SetupCRs(
                    patient,
                    reqs[i].ToString().IndexOf(IRR) != -1,
                    reqs[i].ToString().IndexOf(LEU) != -1,
                    reqs[i].ToString().IndexOf(WASRBC) != -1,
                    reqs[i].ToString().IndexOf(WASPLT) != -1,
                    reqs[i].ToString().IndexOf(CMV) != -1,
                    reqs[i].ToString().IndexOf(SC) != -1
                    );


                _patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.EnhancedTechnologist)); // need to recreate the object to account for caching
                results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
                Assert.IsTrue(results.Count == 1);

                IDictionaryEnumerator enumerator = results.GetEnumerator();
                enumerator.MoveNext();

                Common.StrRes.StringResourceItem message = (Common.StrRes.StringResourceItem)enumerator.Key;
                string validate = message.ResString;

                Assert.IsTrue(ValidTRMessage(reqs[i].ToString(), validate, false));
            }
        }


        [Test]
        public void ComponentRequirements_WB_Fail()
        {
            ArrayList reqs = new ArrayList();
            reqs.Add(IRR);					// Test case #1
            reqs.Add(LEU);					// Test case #2
            reqs.Add(WASRBC);				// Test case #3
            reqs.Add(CMV);					// Test case #4
            reqs.Add(SC);					// Test case #5

            reqs.Add(LEU + ", " + IRR);			// Test case #6
            reqs.Add(IRR + ", " + WASRBC);			// Test case #7
            reqs.Add(CMV + ", " + IRR);			// Test case #8
            reqs.Add(SC + ", " + IRR);				// Test case #9
            reqs.Add(WASRBC + ", " + LEU);			// Test case #10
            reqs.Add(LEU + ", " + CMV);			// Test case #11
            reqs.Add(SC + ", " + LEU);				// Test case #12
            reqs.Add(WASRBC + ", " + CMV);			// Test case #13
            reqs.Add(WASRBC + ", " + SC);			// Test case #14
            reqs.Add(CMV + ", " + SC);				// Test case #15

            reqs.Add(LEU + ", " + IRR + ", " + WASRBC);	// Test case #16
            reqs.Add(LEU + ", " + CMV + ", " + IRR);		// Test case #17
            reqs.Add(SC + ", " + IRR + ", " + LEU);		// Test case #18
            reqs.Add(CMV + ", " + WASRBC + ", " + IRR);	// Test case #19
            reqs.Add(WASRBC + ", " + SC + ", " + IRR);	// Test case #20
            reqs.Add(CMV + ", " + SC + ", " + IRR);		// Test case #21
            reqs.Add(LEU + ", " + CMV + ", " + SC);		// Test case #22
            reqs.Add(SC + ", " + WASRBC + ", " + CMV);	// Test case #23
            reqs.Add(CMV + ", " + WASRBC + ", " + LEU);	// Test case #24
            reqs.Add(WASRBC + ", " + SC + ", " + LEU);	// Test case #25

            reqs.Add(WASRBC + ", " + CMV + ", " + IRR + ", " + LEU);		// Test case #26
            reqs.Add(SC + ", " + WASRBC + ", " + IRR + ", " + LEU);		// Test case #27
            reqs.Add(IRR + ", " + SC + ", " + CMV + ", " + LEU);			// Test case #28
            reqs.Add(SC + ", " + CMV + ", " + IRR + ", " + WASRBC);		// Test case #29
            reqs.Add(SC + ", " + WASRBC + ", " + LEU + ", " + CMV);		// Test case #30

            reqs.Add(WASRBC + ", " + CMV + ", " + IRR + ", " + SC + ", " + LEU);	// Test case #31



            BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E001", "E6043", Common.ABO.A, Common.RH.Negative, Common.ComponentClass.RBC);
            Patient patient = new Patient(_patientGuid);
            this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
            _patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));


            // No TRs here
            Hashtable results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
            Assert.IsTrue(results.Count == 0);


            // First, do the elevated security level
            for (int i = 0; i < reqs.Count; i++)
            {
                this.SetupCRs(
                    patient,
                    reqs[i].ToString().IndexOf(IRR) != -1,
                    reqs[i].ToString().IndexOf(LEU) != -1,
                    reqs[i].ToString().IndexOf(WASRBC) != -1,
                    reqs[i].ToString().IndexOf(WASPLT) != -1,
                    reqs[i].ToString().IndexOf(CMV) != -1,
                    reqs[i].ToString().IndexOf(SC) != -1
                    );


                _patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist)); // need to recreate the object to account for caching
                results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
                Assert.IsTrue(results.Count == 1);

                IDictionaryEnumerator enumerator = results.GetEnumerator();
                enumerator.MoveNext();

                Common.StrRes.StringResourceItem message = (Common.StrRes.StringResourceItem)enumerator.Key;

                Assert.IsTrue(ValidTRMessage(reqs[i].ToString(), message.ResString, true));
            }

            // Next, do the lower security level
            for (int i = 0; i < reqs.Count; i++)
            {
                this.SetupCRs(
                    patient,
                    reqs[i].ToString().IndexOf(IRR) != -1,
                    reqs[i].ToString().IndexOf(LEU) != -1,
                    reqs[i].ToString().IndexOf(WASRBC) != -1,
                    reqs[i].ToString().IndexOf(WASPLT) != -1,
                    reqs[i].ToString().IndexOf(CMV) != -1,
                    reqs[i].ToString().IndexOf(SC) != -1
                    );


                _patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.EnhancedTechnologist)); // need to recreate the object to account for caching
                results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
                Assert.IsTrue(results.Count == 1);

                IDictionaryEnumerator enumerator = results.GetEnumerator();
                enumerator.MoveNext();

                Common.StrRes.StringResourceItem message = (Common.StrRes.StringResourceItem)enumerator.Key;
                string validate = message.ResString;

                Assert.IsTrue(ValidTRMessage(reqs[i].ToString(), validate, false));
            }
        }


        [Test]
        public void ComponentRequirements_PLT_Fail()
        {
            ArrayList reqs = new ArrayList();
            reqs.Add(IRR);					// Test case #1
            reqs.Add(LEU);					// Test case #2
            reqs.Add(WASPLT);				// Test case #3
            reqs.Add(CMV);					// Test case #4

            reqs.Add(LEU + ", " + IRR);			// Test case #5
            reqs.Add(IRR + ", " + WASPLT);			// Test case #6
            reqs.Add(CMV + ", " + IRR);			// Test case #7
            reqs.Add(WASPLT + ", " + LEU);			// Test case #8
            reqs.Add(LEU + ", " + CMV);			// Test case #9
            reqs.Add(WASPLT + ", " + CMV);			// Test case #10

            reqs.Add(LEU + ", " + IRR + ", " + WASPLT);	// Test case #11
            reqs.Add(LEU + ", " + CMV + ", " + IRR);		// Test case #12
            reqs.Add(CMV + ", " + WASPLT + ", " + IRR);	// Test case #13		
            reqs.Add(CMV + ", " + WASPLT + ", " + LEU);	// Test case #14

            reqs.Add(WASPLT + ", " + CMV + ", " + IRR + ", " + LEU);		// Test case #15


            BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E020", "E5464", Common.ABO.O, Common.RH.Negative, Common.ComponentClass.PLT);
            Patient patient = new Patient(_patientGuid);
            this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
            _patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));


            // No TRs here
            Hashtable results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
            Assert.IsTrue(results.Count == 0);


            // First, do the elevated security level
            for (int i = 0; i < reqs.Count; i++)
            {
                this.SetupCRs(
                    patient,
                    reqs[i].ToString().IndexOf(IRR) != -1,
                    reqs[i].ToString().IndexOf(LEU) != -1,
                    reqs[i].ToString().IndexOf(WASRBC) != -1,
                    reqs[i].ToString().IndexOf(WASPLT) != -1,
                    reqs[i].ToString().IndexOf(CMV) != -1,
                    reqs[i].ToString().IndexOf(SC) != -1
                    );


                _patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist)); // need to recreate the object to account for caching
                results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
                Assert.IsTrue(results.Count == 1);

                IDictionaryEnumerator enumerator = results.GetEnumerator();
                enumerator.MoveNext();

                Common.StrRes.StringResourceItem message = (Common.StrRes.StringResourceItem)enumerator.Key;

                Assert.IsTrue(ValidTRMessage(reqs[i].ToString(), message.ResString, true));
            }

            // Next, do the lower security level
            for (int i = 0; i < reqs.Count; i++)
            {
                this.SetupCRs(
                    patient,
                    reqs[i].ToString().IndexOf(IRR) != -1,
                    reqs[i].ToString().IndexOf(LEU) != -1,
                    reqs[i].ToString().IndexOf(WASRBC) != -1,
                    reqs[i].ToString().IndexOf(WASPLT) != -1,
                    reqs[i].ToString().IndexOf(CMV) != -1,
                    reqs[i].ToString().IndexOf(SC) != -1
                    );


                _patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.EnhancedTechnologist)); // need to recreate the object to account for caching
                results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
                Assert.IsTrue(results.Count == 1);

                IDictionaryEnumerator enumerator = results.GetEnumerator();
                enumerator.MoveNext();

                Common.StrRes.StringResourceItem message = (Common.StrRes.StringResourceItem)enumerator.Key;
                string validate = message.ResString;

                Assert.IsTrue(ValidTRMessage(reqs[i].ToString(), validate, false));
            }
        }


        [Test]
        public void ComponentRequirements_OTHER_Fail()
        {
            ArrayList reqs = new ArrayList();
            reqs.Add(IRR);					// Test case #1
            reqs.Add(CMV);					// Test case #2

            reqs.Add(CMV + ", " + IRR);			// Test case #3



            BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E040", "E3805", Common.ABO.O, Common.RH.Negative, Common.ComponentClass.Other);
            Patient patient = new Patient(_patientGuid);
            this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
            _patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));


            // No TRs here
            Hashtable results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
            Assert.IsTrue(results.Count == 0);


            // First, do the elevated security level
            for (int i = 0; i < reqs.Count; i++)
            {
                this.SetupCRs(
                    patient,
                    reqs[i].ToString().IndexOf(IRR) != -1,
                    reqs[i].ToString().IndexOf(LEU) != -1,
                    reqs[i].ToString().IndexOf(WASRBC) != -1,
                    reqs[i].ToString().IndexOf(WASPLT) != -1,
                    reqs[i].ToString().IndexOf(CMV) != -1,
                    reqs[i].ToString().IndexOf(SC) != -1
                    );


                _patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist)); // need to recreate the object to account for caching
                results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
                Assert.IsTrue(results.Count == 1);

                IDictionaryEnumerator enumerator = results.GetEnumerator();
                enumerator.MoveNext();

                Common.StrRes.StringResourceItem message = (Common.StrRes.StringResourceItem)enumerator.Key;

                Assert.IsTrue(ValidTRMessage(reqs[i].ToString(), message.ResString, true));
            }

            // Next, do the lower security level
            for (int i = 0; i < reqs.Count; i++)
            {
                this.SetupCRs(
                    patient,
                    reqs[i].ToString().IndexOf(IRR) != -1,
                    reqs[i].ToString().IndexOf(LEU) != -1,
                    reqs[i].ToString().IndexOf(WASRBC) != -1,
                    reqs[i].ToString().IndexOf(WASPLT) != -1,
                    reqs[i].ToString().IndexOf(CMV) != -1,
                    reqs[i].ToString().IndexOf(SC) != -1
                    );


                _patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.EnhancedTechnologist)); // need to recreate the object to account for caching
                results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
                Assert.IsTrue(results.Count == 1);

                IDictionaryEnumerator enumerator = results.GetEnumerator();
                enumerator.MoveNext();

                Common.StrRes.StringResourceItem message = (Common.StrRes.StringResourceItem)enumerator.Key;
                string validate = message.ResString;

                Assert.IsTrue(ValidTRMessage(reqs[i].ToString(), validate, false));
            }
        }





		#region re-engineered

		[Test]
		public void UnitSelected_Pass()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			// ToDo: must call delete test data when finished
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			_patientUnitIssue.AddUnit(bloodUnit, false);
			Assert.IsTrue(_patientUnitIssue.UnitSelected(bloodUnit.BloodUnitGuid));
		}
		[Test]
		public void UnitSelected_Fail()
		{
			Assert.IsTrue(!_patientUnitIssue.UnitSelected(Guid.NewGuid()));
		}

		
		[Test]
		public void UnitsAvailable_Pass()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			// ToDo: must call delete test data when finished
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(_patientUnitIssue.UnitsAvailable());
		}
		[Test]
		public void UnitsAvailable_Fail()
		{
			Assert.IsTrue(!_patientUnitIssue.UnitsAvailable());
		}


		[Test]
		public void GetUserNameList_Pass()
		{
			ArrayList users = _patientUnitIssue.GetUserNameList();
			Assert.IsTrue(users != null && users.Count > 0);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetUserNameList_Fail()
		{
		}


		[Test]
		public void GetLocations_Pass()
		{
			ArrayList locations = _patientUnitIssue.GetLocations();
			Assert.IsTrue(locations != null && locations.Count > 0);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetLocations_Fail()
		{
		}


		[Test]
		public void GetOrderingClinicians_Pass()
		{
			// Must set up data so there are some units issueable
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			
			// ToDo: must call delete test data when finished
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			
			ArrayList clinicians = _patientUnitIssue.GetOrderingClinicians();
			Assert.IsTrue(clinicians != null && clinicians.Count > 0);
		}
		[Test]
		public void GetOrderingClinicians_Fail()
		{
			Assert.IsTrue(_patientUnitIssue.GetOrderingClinicians().Count == 0);
		}


		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void AddIssuedUnitComment_Pass()
		{
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void AddIssuedUnitComment_Fail()
		{
		}


		[Test]
		public void GetMessagesForTooltip_Pass()
		{
			// Must set up data so there are some units issueable
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			
			// ToDo: must call delete test data when finished
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			
			string messages = _patientUnitIssue.GetMessagesForTooltip(bloodUnit.BloodUnitGuid);
			Assert.IsTrue(messages != null && messages == string.Empty);
		}
		[Test]
		public void GetMessagesForTooltip_Fail()
		{
			// Must set up data so there are some units issueable
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.B, Common.RH.Positive, Common.ComponentClass.RBC);
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET UnitExpirationDate = '" + DateTime.Now.AddDays(-2) + "' WHERE BloodUnitGuid = '" + bloodUnit.BloodUnitGuid + "'");

			// ToDo: must call delete test data when finished
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			
			string messages = _patientUnitIssue.GetMessagesForTooltip(bloodUnit.BloodUnitGuid);
			Assert.IsTrue(messages != null && messages != string.Empty);
		}

        [Test]
        public void GetMessagesForTooltip_CodeCoverage()
        {
            Guid myPatient = CreatePatient();
            BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
            this.CreateTestData(myPatient, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);

            //***Unit expired, user is powerful enough
            UnitTestUtil.RunSQL("update bloodunitmedia set unitexpirationdate = DATEADD(DAY, -1, GETUTCDATE())  WHERE bloodunitguid = '" + bloodUnit.BloodUnitGuid.ToString() + "'");
            BOL.PatientUnitIssue myPUI1 = new BOL.PatientUnitIssue(new BOL.Patient(myPatient), GetVbecsUser(Common.UserRoleType.EnhancedTechnologist));

            string messages1 = myPUI1.GetMessagesForTooltip(bloodUnit.BloodUnitGuid);
            Assert.IsTrue(messages1.Contains(Common.StrRes.ConfMsg.UC073.UnitExpired().ResString), "error1 message");
            Assert.IsTrue(myPUI1.UnitSelectable(bloodUnit.BloodUnitGuid), "expired unit selectable");

            //***Unit expired, user is too tiny
            BOL.PatientUnitIssue myPUIb = new BOL.PatientUnitIssue(new BOL.Patient(myPatient), GetVbecsUser(Common.UserRoleType.Technologist));

            string messagesb = myPUIb.GetMessagesForTooltip(bloodUnit.BloodUnitGuid);
            Assert.IsTrue(messagesb.Contains(Common.StrRes.ConfMsg.UC073.UnitExpired().ResString), "errorb message");
            Assert.IsFalse(myPUIb.UnitSelectable(bloodUnit.BloodUnitGuid), "expired unit not selectable");


            //***Specimen expires before unit, user is too tiny
            UnitTestUtil.RunSQL("update patientspecimen set SpecimenExpirationDate = DATEADD(DAY, -4, SpecimenExpirationDate)  WHERE patientguid = '" + myPatient.ToString() + "'");
            BOL.PatientUnitIssue myPUI2 = new BOL.PatientUnitIssue(new BOL.Patient(myPatient), GetVbecsUser(Common.UserRoleType.Technologist));
            string messages2 = myPUI2.GetMessagesForTooltip(bloodUnit.BloodUnitGuid);
            Assert.IsTrue(messages2.Contains(Common.StrRes.ConfMsg.UC073.SpecimenExpired().ResString), "error2 message");
            Assert.IsFalse(myPUI2.UnitSelectable(bloodUnit.BloodUnitGuid), "expired specimen not selectable");

            //***Specimen expires before unit, user is powerful enough
            BOL.PatientUnitIssue myPUI3 = new BOL.PatientUnitIssue(new BOL.Patient(myPatient), GetVbecsUser(Common.UserRoleType.EnhancedTechnologist));

            string messages3 = myPUI3.GetMessagesForTooltip(bloodUnit.BloodUnitGuid);
            Assert.IsTrue(messages3.Contains(Common.StrRes.ConfMsg.UC073.SpecimenExpired().ResString), "error3 message");
            Assert.IsTrue(myPUI3.UnitSelectable(bloodUnit.BloodUnitGuid), "expired specimen selectable");

        }


		[Test]
		public void UnitSelectable_Pass()
		{
			// Must set up data so there are some units issueable
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			
			// ToDo: must call delete test data when finished
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			
			bool selectable = _patientUnitIssue.UnitSelectable(bloodUnit.BloodUnitGuid);
			// It's impossible to determine whether or not a random unit will be selecTABLES.
			Assert.IsTrue(selectable);
		}
		[Test]
		public void UnitSelectable_Fail()
		{
			// Must set up data so there are some units issueable
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			//UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET UnitExpirationDate = '" + DateTime.Now.AddDays(-2) + "'");
			UnitTestUtil.RunSQL("UPDATE BloodUnitStatus SET QuarantineIndicator = '1' WHERE BloodUnitGuid = '" + bloodUnit.BloodUnitGuid + "'");
			
			// ToDo: must call delete test data when finished
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			
			// known unit
			bool selectable = _patientUnitIssue.UnitSelectable(bloodUnit.BloodUnitGuid);
			// It's impossible to determine whether or not a random unit will be selecTABLES.
			Assert.IsTrue(!selectable);

			// random 
			Assert.IsTrue(!_patientUnitIssue.UnitSelectable(Guid.NewGuid()));
		}


		[Test]
		public void AddUnit_Pass()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			
			// ToDo: must call delete test data when finished
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			_patientUnitIssue.RemoteStorageIndicator = true;
			Assert.IsTrue(_patientUnitIssue.AddUnit(bloodUnit, false));
		}
		[Test]
		public void AddUnit_Pass_SelectionOrder()
		{
			// Setup
			BOL.BloodUnit bloodUnitAutologous = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			BOL.BloodUnit bloodUnitDirected = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			BOL.BloodUnit bloodUnitDirectedTwo = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			Guid [] bloodUnitGuids = {bloodUnitAutologous.BloodUnitGuid, bloodUnitDirected.BloodUnitGuid, bloodUnitDirectedTwo.BloodUnitGuid};  


			#region Make units directed/ autologous

			// AutologousPut in transaction
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");
			string donate = "UPDATE BloodUnit SET DonationTypeId = " + (int) Common.DonationType.ForAutologousUseOnly + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'";
			UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 10 + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");

			// Directed
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");
			UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 14 + " WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");

			// Regenerate BUs so they pick up the update donation codes
			bloodUnitAutologous = new BOL.BloodUnit(bloodUnitAutologous.BloodUnitGuid);
			bloodUnitDirected = new BOL.BloodUnit(bloodUnitDirected.BloodUnitGuid);
			bloodUnitDirectedTwo = new BOL.BloodUnit(bloodUnitDirected.BloodUnitGuid);

			#endregion

			// ToDo: must call delete test data when finished
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnitGuids);

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			
			_patientUnitIssue.AddUnit(bloodUnitDirected, false);
			_patientUnitIssue.AddUnit(bloodUnitAutologous, false);
			
			Assert.IsTrue(_patientUnitIssue.AddUnit(bloodUnitDirected, false));
		}
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void AddUnit_Fail()
		{
			_patientUnitIssue.AddUnit(null, false);
		}
		[Test]
		public void AddUnit_Fail_SelectionOrder()
		{
			// Setup
			BOL.BloodUnit bloodUnitAutologous = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			BOL.BloodUnit bloodUnitDirected = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			Guid [] bloodUnitGuids = {bloodUnitAutologous.BloodUnitGuid, bloodUnitDirected.BloodUnitGuid};  


			#region Make units directed/ autologous

			// AutologousPut in transaction
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");
			string donate = "UPDATE BloodUnit SET DonationTypeId = " + (int) Common.DonationType.ForAutologousUseOnly + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'";
			UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 10 + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");

			// Directed
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");
			UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 14 + " WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");

			// Regenerate BUs so they pick up the update donation codes
			bloodUnitAutologous = new BOL.BloodUnit(bloodUnitAutologous.BloodUnitGuid);
			bloodUnitDirected = new BOL.BloodUnit(bloodUnitDirected.BloodUnitGuid);

			#endregion

			// ToDo: must call delete test data when finished
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnitGuids);

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.AddUnit(bloodUnitDirected, false));
		}
		[Test]
		public void AddUnit_Fail_ExpirationDate_SecurityLevel1()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			
			// ToDo: must call delete test data when finished
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			bloodUnit.BloodUnitMedia.UnitExpirationDate = _patientUnitIssue.IssueDate.AddDays(-1);
			Assert.IsTrue(!_patientUnitIssue.AddUnit(bloodUnit, false));
		}
		[Test]
		public void AddUnit_Fail_ExpirationDate_SecurityLevel2()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			
			// ToDo: must call delete test data when finished
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.EnhancedTechnologist));
			bloodUnit.BloodUnitMedia.UnitExpirationDate = _patientUnitIssue.IssueDate.AddDays(-1);
			Assert.IsTrue(!_patientUnitIssue.AddUnit(bloodUnit, false));
		}
		[Test]
		public void AddUnit_Fail_SpecimenExpirationDate_SecurityLevel1()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			_patientUnitIssue.IssueDate = _patientUnitIssue.GetSpecimenAssociatedWithUnit(bloodUnit.BloodUnitGuid).SpecimenExpirationDate.AddDays(1);
			Assert.IsTrue(!_patientUnitIssue.AddUnit(bloodUnit, false));
		}
		[Test]
		public void AddUnit_Fail_SpecimenExpirationDate_SecurityLevel2()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.EnhancedTechnologist));
			_patientUnitIssue.IssueDate = _patientUnitIssue.GetSpecimenAssociatedWithUnit(bloodUnit.BloodUnitGuid).SpecimenExpirationDate.AddDays(1);
			Assert.IsTrue(!_patientUnitIssue.AddUnit(bloodUnit, false));
		}


		[Test]
		public void RemoveUnit_Pass_Satisfactory()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			_patientUnitIssue.AddUnit((BOL.BloodUnit) _patientUnitIssue.Units[0], true);
			_patientUnitIssue.RemoveUnit((BOL.BloodUnit) _patientUnitIssue.Units[0], false);

			Assert.IsTrue(!_patientUnitIssue.Units.Contains(bloodUnit));
		}
		[Test]
		public void RemoveUnit_Pass_Unsatisfactory()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			_patientUnitIssue.AddUnit( (BOL.BloodUnit) _patientUnitIssue.Units[0], false);
			
			Assert.IsTrue(_patientUnitIssue.RemoveUnit((BOL.BloodUnit) _patientUnitIssue.Units[0], true));
		}
		[Test]
		public void RemoveUnit_Pass_Unsatisfactory_EI()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			_patientUnitIssue.AddUnit( (BOL.BloodUnit) _patientUnitIssue.Units[0], true);
			
			Assert.IsTrue(_patientUnitIssue.RemoveUnit((BOL.BloodUnit) _patientUnitIssue.Units[0], true));
		}
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void RemoveUnit_Fail()
		{
			_patientUnitIssue.RemoveUnit(null, false);
		}


		[Test]
		public void Save_Pass()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			_patientUnitIssue.AddUnit(bloodUnit, false);
			
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.IssueUnit);
			Assert.IsTrue(_patientUnitIssue.Save(Common.WorkloadProcessID.IssueUnit));
			UnitTestUtil.RunSQL("DELETE FROM IssuedUnit WHERE " + TABLES.IssuedUnit.BloodUnitGuid + " = '" + bloodUnit.BloodUnitGuid + "'");
		}
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void Save_Fail()
		{
			_patientUnitIssue.Save(Common.WorkloadProcessID.Unknown);
		}


		[Test]
		public void GetSpecimenAssociatedWithUnit_Pass()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			_patientUnitIssue.AddUnit(bloodUnit, false);
		
			Assert.IsTrue( ((BOL.Specimen) _patientUnitIssue.GetSpecimenAssociatedWithUnit(bloodUnit.BloodUnitGuid)) != null );
		}
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void GetSpecimenAssociatedWithUnit_EmptyGuid_Pass()
		{
			Assert.IsTrue(_patientUnitIssue.GetSpecimenAssociatedWithUnit(Guid.Empty) == null );
		}
		[Test]
		public void GetSpecimenAssociatedWithUnit_BadGuid_Fail()
		{
			Assert.IsTrue(_patientUnitIssue.GetSpecimenAssociatedWithUnit(Guid.NewGuid()) == null );
		}


		[Test]
		public void GetOrderedUnitAssociatedWithUnit_Pass()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			_patientUnitIssue.AddUnit(bloodUnit, false);
		
			Assert.IsTrue( ((BOL.OrderedUnit) _patientUnitIssue.GetOrderedUnitAssociatedWithUnit(bloodUnit.BloodUnitGuid)) != null );
		}
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void GetOrderedUnitAssociatedWithUnit_EmptyGuid_Fail()
		{
			Assert.IsTrue(_patientUnitIssue.GetOrderedUnitAssociatedWithUnit(Guid.Empty) == null );
		}
		[Test]
		public void GetOrderedUnitAssociatedWithUnit_BadGuid_Fail()
		{
			Assert.IsTrue(_patientUnitIssue.GetOrderedUnitAssociatedWithUnit(Guid.NewGuid()) == null );
		}


		[Test]
		public void IsEmergencyIssue_Pass()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.O, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", true, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
	
			Assert.IsTrue( _patientUnitIssue.IsEmergencyIssue(bloodUnit.BloodUnitGuid));
		}
		[Test]
		public void IsEmergencyIssue_Fail()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
		
			Assert.IsTrue( !_patientUnitIssue.IsEmergencyIssue(bloodUnit.BloodUnitGuid));
		}

		
		[Test]
		public void ValidateUnit_Pass()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			
			// need to refresh to get unit status
			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);
			
			Hashtable results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			Assert.IsTrue(results.Count == 0);


			// Now, test Caution Tag: 1
			BOL.CautionTag ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(ct.PrintOK && ct.Message == string.Empty, "CautionTag: Print OK");
		}
		[Test]
		public void ValidateUnit_Pass_Two()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.O, Common.RH.Negative, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			
			// need to refresh to get unit status
			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);
						
			Hashtable results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, null);
			Assert.IsTrue(results.Count == 1);	
			//Hashtable no longer returns bool values 2/11/2006
			//&&		(bool) results[Common.StrRes.ConfMsg.UC073.EmergencyIssue()]);

			// Now, test Caution Tag: 2
			BOL.CautionTag ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			// Empty string here and not above because rules for EI message different
			Assert.IsTrue(ct.PrintOK && ct.Message == string.Empty, "CautionTag: OK");
		}
		[Test]
		public void ValidateUnit_Pass_FFP()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E012", "E4507", Common.ABO.AB, Common.RH.Negative, Common.ComponentClass.FFP);
			this.CreateTestData(_patientGuid, Common.ComponentClass.FFP, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
	
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			
			// need to refresh to get unit status
			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);
			
			Hashtable results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, null);
			Assert.IsTrue(results.Count == 1);

			// Now, test Caution Tag: 3
			BOL.CautionTag ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(!ct.PrintOK && ct.Message == Common.StrRes.InfoMsg.UC043.UnitFrozen().ResString, "CautionTag: Frozen");
		}
		[Test]
		public void ValidateUnit_EI_NoResults()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.O, Common.RH.Negative, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE PatientSpecimenGuid IN ('" + _specimenOneGuid + "','" + _specimenTwoGuid + "')");

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			Hashtable results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			Assert.IsTrue(results.Count == 1);

			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);

			// Now, test Caution Tag: 4
			BOL.CautionTag ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(ct.PrintOK && ct.Message == Common.StrRes.InfoMsg.UC043.EmergencyIssue().ResString, "CautionTag: EI");		
		}

		[Test]
		public void ValidateUnit_EI_ConflictingResults()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.O, Common.RH.Negative, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
	
			Guid testGuid = this.SetupSpecimenTest(_specimenTwoGuid, _orderedTestGuid, "18", "B");
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			Hashtable results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			Assert.IsTrue(results.Count == 1);

			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);

			// Now, test Caution Tag: 5
			BOL.CautionTag ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(ct.PrintOK && ct.Message == Common.StrRes.InfoMsg.UC043.EmergencyIssue().ResString, "CautionTag: EI");

			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid = '" + testGuid + "'");
		}


		[Test]
		public void ValidateUnit_Fail_No_Override_Part_One()
		{
			// BR_73.23: Specimen testing not complete
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", true, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			Hashtable results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
	
			Assert.IsTrue(results.Count == 1 && 
				results.ContainsKey(Common.StrRes.InfoMsg.UC073.SpecimenTestingNotCompleted()));
			//Hashtable no longer returns bool values 2/11/2006
			//&&		(bool) results[Common.StrRes.InfoMsg.UC073.SpecimenTestingNotCompleted()] == false);

			// need to refresh to get unit status
			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);

			// Now, test Caution Tag: 6
			BOL.CautionTag ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(!ct.PrintOK && ct.Message == Common.StrRes.InfoMsg.UC043.EmergencyIssueInvalid().ResString, "CautionTag: No EI");



			// BR_3.05: No results on specimen
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE PatientSpecimenGuid IN ('" + _specimenOneGuid + "','" + _specimenTwoGuid + "')");
			// must refresh for caching
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));

			
			results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			Assert.IsTrue(results.Count == 3 && 
				results.ContainsKey(Common.StrRes.InfoMsg.UC073.NoABORhResultsOnSpecimen()) && 
				(Common.ExceptionType) results[Common.StrRes.InfoMsg.UC073.NoABORhResultsOnSpecimen()] == Common.ExceptionType.Unissuable);


			// Now, test Caution Tag: 7
			ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(!ct.PrintOK && ct.Message == Common.StrRes.InfoMsg.UC043.EmergencyIssueInvalid().ResString, "CautionTag: No EI 2");



			// BR_73.23: No crossmatch results
			UnitTestUtil.RunSQL("UPDATE OrderedUnit SET CrossmatchResultCode = NULL WHERE OrderedUnitGuid = '" + _orderedUnitGuid + "'");
			results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			Assert.IsTrue(results.Count > 0 && 
				results.ContainsKey(Common.StrRes.InfoMsg.UC073.NoCrossmatchResults()) && 
				(Common.ExceptionType) results[Common.StrRes.InfoMsg.UC073.NoCrossmatchResults()] == Common.ExceptionType.Unissuable);


			// Now, test Caution Tag: 8
			ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(!ct.PrintOK && ct.Message == Common.StrRes.InfoMsg.UC043.EmergencyIssueInvalid().ResString, "CautionTag: No EI 3");


	
			// BR_73.35: Inconclusive crossmatch
			UnitTestUtil.RunSQL("UPDATE OrderedUnit SET CrossmatchResultCode = 'Z' WHERE OrderedUnitGuid = '" + _orderedUnitGuid + "'");
			results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			Assert.IsTrue(results.Count > 0 && 
				results.ContainsKey(Common.StrRes.InfoMsg.UC073.InconclusiveCrossmatch()) && 
				(Common.ExceptionType) results[Common.StrRes.InfoMsg.UC073.InconclusiveCrossmatch()] == Common.ExceptionType.Unissuable);

			// Now, test Caution Tag: 9
			ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(!ct.PrintOK && ct.Message == Common.StrRes.InfoMsg.UC043.InconclusiveCrossmatch().ResString, "CautionTag: Inconclusive XM");

		}


		[Test]
		public void ValidateUnit_Fail_No_Override_Part_Two()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E001", "E0001", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.WB);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
		
			// need to refresh to get unit status
			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);
			//Thread.Sleep(121000);

			// Setup inconsistent tests
			Guid testGuid = this.SetupSpecimenTest(_specimenOneGuid, _orderedTestGuid, "18", "B");
			Guid testTwoGuid = this.SetupSpecimenTest(_specimenOneGuid, _orderedTestGuid, "21", "P");
			//UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE PatientSpecimenGuid = '" + _specimenTwoGuid + "' AND BloodTestTypeId = '21'");
				
			// Repeat discrepancy (both repeat and with another specimen)
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			Hashtable results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));

			Assert.IsTrue(results.ContainsKey(Common.StrRes.InfoMsg.UC073.ABORhDiscrepancy()), "Test #1: ABO Discrepancy");
	
	
			// Now, test Caution Tag: 10
			BOL.CautionTag ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(!ct.PrintOK && ct.Message == Common.StrRes.InfoMsg.UC043.EmergencyIssueInvalid().ResString, "Test #2: CautionTag: No EI 4");




			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid = '" + testGuid + "'");
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid = '" + testTwoGuid + "'");
			UnitTestUtil.RunSQL("DELETE FROM BloodUnitTest WHERE BloodUnitGuid = '" + bloodUnit.BloodUnitGuid + "'");



			// repeat whole thing with EI unit
			bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.O, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
		
			// need to refresh to get unit status
			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);
			//Thread.Sleep(120000);

			// Setup inconsistent tests
			testGuid = this.SetupSpecimenTest(_specimenOneGuid, _orderedTestGuid, "18", "B");
			testTwoGuid = this.SetupSpecimenTest(_specimenOneGuid, _orderedTestGuid, "21", "P");
				
			// Repeat discrepancy (both repeat and with another specimen)
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
	
			Assert.IsTrue(results.Count == 1, "Test #3:");


			// Now, test Caution Tag: 11
			ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(ct.PrintOK && ct.Message == Common.StrRes.InfoMsg.UC043.EmergencyIssue().ResString, "Test #4: CautionTag: EI (inconsistent tests)");

			// null specimen
			results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, null);
			Assert.IsTrue(results.Count == 1, "Test #5: ");


			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid = '" + testGuid + "'");
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid = '" + testTwoGuid + "'");		
			UnitTestUtil.RunSQL("DELETE FROM BloodUnitTest WHERE BloodUnitGuid = '" + bloodUnit.BloodUnitGuid + "'");			


			//PART TWO ---- Keeps going, and going, and going....
			bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
		
			//Thread.Sleep(120000);

			// Setup inconsistent tests
			testGuid = this.SetupSpecimenTest(_specimenTwoGuid, _orderedTestGuid, "18", "B");
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE PatientSpecimenGuid = '" + _specimenTwoGuid + "' AND BloodTestTypeId = '21'");
	
	
			// Repeat discrepancy (both repeat and with another specimen)
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			Assert.IsTrue(results.Count > 0, "Test #6");


			// Now, test Caution Tag: 13
			ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(!ct.PrintOK, "Test #7: CautionTag: EI invalid" );


			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid = '" + testGuid + "'");
			UnitTestUtil.RunSQL("DELETE FROM BloodUnitTest WHERE BloodUnitGuid = '" + bloodUnit.BloodUnitGuid + "'");
		}


		[Test]
		public void ValidateUnit_Fail_No_Override_Part_Three()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));

			// need to refresh to get unit status
			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);

			// Set data - Quarantined unit
			bloodUnit.BloodUnitStatus.QuarantineIndicator = true;

			Hashtable results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			Assert.IsTrue(results.Count == 1 &&
				results.ContainsKey(Common.StrRes.InfoMsg.UC073.UnitQuarantined())
				&& (Common.ExceptionType) results[Common.StrRes.InfoMsg.UC073.UnitQuarantined()] == Common.ExceptionType.Unissuable, "Quarantined");



			// Now, test Caution Tag: 14
			BOL.CautionTag ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(!ct.PrintOK && ct.Message == Common.StrRes.InfoMsg.UC043.UnitQuarantined().ResString, "CautionTag: Quarantine");

			// Reset
			bloodUnit.BloodUnitStatus.QuarantineIndicator = false;


			// Set data - Incompatible crossmatch
			UnitTestUtil.RunSQL("UPDATE OrderedUnit SET CrossmatchResultCode = 'I' WHERE OrderedUnitGuid = '" + _orderedUnitGuid + "'");

			results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			Assert.IsTrue(results.Count == 1 &&
				results.ContainsKey(Common.StrRes.InfoMsg.UC073.UnitCrossmatchBad())
				&& (Common.ExceptionType) results[Common.StrRes.InfoMsg.UC073.UnitCrossmatchBad()] == Common.ExceptionType.Unissuable, "Incompatible crossmatch");


			// Now, test Caution Tag: 15
			ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(!ct.PrintOK && ct.Message == Common.StrRes.InfoMsg.UC043.IncompatibleCrossmatchResults().ResString, "CautionTag: Frozen");

			// Reset
			UnitTestUtil.RunSQL("UPDATE OrderedUnit SET CrossmatchResultCode = 'C' WHERE OrderedUnitGuid = '" + _orderedUnitGuid + "'");



			// Set data - Frozen
			bloodUnit.ProductType.ProductTypeCode = "E004";

			results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			Assert.IsTrue(results.Count == 1 &&
				results.ContainsKey(Common.StrRes.InfoMsg.UC073.UnitFrozen())
				&& (Common.ExceptionType) results[Common.StrRes.InfoMsg.UC073.UnitFrozen()] == Common.ExceptionType.Unissuable, "Frozen");
		}


		[Test]
		public void ValidateUnit_Fail_No_Override_Insufficient_Security()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			
			// Antigen Neg requirements
			// setup failure data
			this.SetupAntigenTr(new BOL.Patient(_patientGuid), bloodUnit.BloodUnitGuid, true);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));

			// need to refresh to get unit status
			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);
	
			Hashtable results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			Assert.IsTrue(results.Count == 1 &&
				results.ContainsKey(Common.StrRes.InfoMsg.UC073.AntibodyProblem("Anti-C", bloodUnit.EyeReadableUnitId, "C"))
				&& (Common.ExceptionType) results[Common.StrRes.InfoMsg.UC073.AntibodyProblem("Anti-C", bloodUnit.EyeReadableUnitId, "C")] == Common.ExceptionType.Unissuable);



			// Now, test Caution Tag: 16
			BOL.CautionTag ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(!ct.PrintOK && ct.Message == Common.StrRes.InfoMsg.UC043.AntibodyProblemNoOverride("Anti-C", bloodUnit.EyeReadableUnitId, "C").ResString, "CautionTag: Antibody");


			// Reset
			UnitTestUtil.RunSQL("DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + _patientGuid + "'");



			// Antigen Neg requirements
			// setup failure data
			this.SetupAntigenTr(new BOL.Patient(_patientGuid), bloodUnit.BloodUnitGuid, false);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));

			results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			Assert.IsTrue(results.Count == 1 &&
				results.ContainsKey(Common.StrRes.InfoMsg.UC073.AntigenProblem("C", bloodUnit.EyeReadableUnitId))
				&& (Common.ExceptionType) results[Common.StrRes.InfoMsg.UC073.AntigenProblem("C", bloodUnit.EyeReadableUnitId)] == Common.ExceptionType.Unissuable);



			// Now, test Caution Tag: 17
			ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(!ct.PrintOK && ct.Message == Common.StrRes.InfoMsg.UC043.AntigenProblemNoOverride("C", bloodUnit.EyeReadableUnitId).ResString, "CautionTag: Antigen");




			// Reset
			UnitTestUtil.RunSQL("DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + _patientGuid + "'");



			// Component Requirements
			// setup failure data
			this.SetupAllCRs(new BOL.Patient(_patientGuid));

			results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			Assert.IsTrue(results.Count == 1
				&& results.ContainsValue(Common.ExceptionType.Unissuable));


			// Now, test Caution Tag: 18
			ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(!ct.PrintOK, "CautionTag: TR");


			// Reset
			UnitTestUtil.RunSQL("DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + _patientGuid + "'");



			//	// Expired specimen
			//	// setup failure data
			//	DateTime oldIssueDate = _patientUnitIssue.IssueDate;
			//    _patientUnitIssue.IssueDate = DateTime.Now.AddDays(50);
			//	results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			//    
			//	Assertion.IsTrue(results.Count == 1 &&
			//		results.ContainsKey(Common.StrRes.InfoMsg.UC073.SpecimenExpired())
			//		&& (Common.ExceptionType) results[Common.StrRes.InfoMsg.UC073.SpecimenExpired()] == Common.ExceptionType.Unissuable);


			// Now, test Caution Tag: 19 ToDo
			ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			//	Assertion.IsTrue("CautionTag: Frozen", !ct.PrintOK && ct.Message == Common.StrRes.InfoMsg.UC043.UnitFrozen().ResString);
			//
			//
			//	// Reset
			//	_patientUnitIssue.IssueDate = oldIssueDate;



			// Unit expiration date
			// setup failure data
			bloodUnit.BloodUnitMedia.UnitExpirationDate = DateTime.Now.AddDays(-50);
			results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
    

			// Now, test Caution Tag: 20
			ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(!ct.PrintOK && ct.Message == Common.StrRes.InfoMsg.UC043.UnitExpiredNoOverride().ResString, "CautionTag: Frozen");



			//	Assertion.IsTrue(results.Count == 1 &&
			//		results.ContainsKey(Common.StrRes.InfoMsg.UC073.UnitExpired())
			//		&& (Common.ExceptionType) results[Common.StrRes.InfoMsg.UC073.UnitExpired()] == Common.ExceptionType.Unissuable);			
		}


		[Test]
		public void ValidateUnit_Fail_Override_Sufficient_Security()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));

			// need to refresh to get unit status
			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);

			// ToDo: Fix when info messages checked back in
			// Antigen Neg requirements
			// setup failure data
			this.SetupAntigenTr(new BOL.Patient(_patientGuid), bloodUnit.BloodUnitGuid, true);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));		

			Hashtable results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			Assert.IsTrue(results.Count == 1 &&
				results.ContainsKey(Common.StrRes.ConfMsg.UC073.AntibodyProblem("Anti-C", bloodUnit.EyeReadableUnitId, "C"))
				&& (Common.ExceptionType) results[Common.StrRes.ConfMsg.UC073.AntibodyProblem("Anti-C", bloodUnit.EyeReadableUnitId, "C")] == Common.ExceptionType.Antigen_positive_untested_units_issued);


			// Now, test Caution Tag: 21
			BOL.CautionTag ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(!ct.PrintOK && ct.Message == Common.StrRes.InfoMsg.UC043.AntibodyProblemNoOverride("Anti-C", bloodUnit.EyeReadableUnitId, "C").ResString, "CautionTag: Antibody");

			// Reset
			UnitTestUtil.RunSQL("DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + _patientGuid + "'");



			this.SetupAntigenTr(new BOL.Patient(_patientGuid), bloodUnit.BloodUnitGuid, false);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));

			results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			Assert.IsTrue(results.Count == 1 &&
				results.ContainsKey(Common.StrRes.ConfMsg.UC073.AntigenProblem("C", bloodUnit.EyeReadableUnitId))
				&& (Common.ExceptionType) results[Common.StrRes.ConfMsg.UC073.AntigenProblem("C", bloodUnit.EyeReadableUnitId)] == Common.ExceptionType.Antigen_positive_untested_units_issued);



			// Now, test Caution Tag: 22, ToDO
			ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist), false);
			Assert.IsTrue(ct.PrintOK, "CautionTag: Antigen problem");



			// Reset
			UnitTestUtil.RunSQL("DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + _patientGuid + "'");



			// Component Requirements
			// setup failure data
			this.SetupAllCRs(new BOL.Patient(_patientGuid));
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));

			results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			Assert.IsTrue(results.Count == 1
				&&	results.ContainsValue(Common.ExceptionType.Unit_issued_with_unsatisfied_transfusion_requirement));


			// Now, test Caution Tag: 23
			ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(!ct.PrintOK, "CautionTag: CRs");


			// Reset
			UnitTestUtil.RunSQL("DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + _patientGuid + "'");



			//		// Expired specimen
			//		// setup failure data
			//		DateTime oldIssueDate = _patientUnitIssue.IssueDate;
			//		_patientUnitIssue.IssueDate = DateTime.Now.AddDays(50);
			//		results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			//	  
			//		//Hashtables don't contain bool values 2/11/06
			//		Assertion.IsTrue(results.Count == 2 &&
			//			results.ContainsKey(Common.StrRes.ConfMsg.UC073.SpecimenExpired()) &&
			//			(Common.ExceptionType) results[Common.StrRes.ConfMsg.UC073.SpecimenExpired()] == Common.ExceptionType.Unit_issued_on_expired_specimen &&
			//			results.ContainsKey(Common.StrRes.ConfMsg.UC073.UnitExpired()) &&	
			//			(Common.ExceptionType) results[Common.StrRes.ConfMsg.UC073.UnitExpired()] == Common.ExceptionType.Issued_expired_unit);
			//
			//
			//		// Now, test Caution Tag: 24
			//		ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			//		Assertion.IsTrue("CautionTag: Expired", ct.PrintOK);
			//
			//
			//		// Reset
			//		_patientUnitIssue.IssueDate = oldIssueDate;



			// Already tested, delete

			// Unit expiration date
			// setup failure data
			bloodUnit.BloodUnitMedia.UnitExpirationDate = DateTime.Now.AddDays(-50);
			results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
	    
			//		Assertion.IsTrue(results.Count == 1 &&
			//			results.ContainsKey(Common.StrRes.ConfMsg.UC073.UnitExpired())
			//			&& (Common.ExceptionType) results[Common.StrRes.ConfMsg.UC073.UnitExpired()] == Common.ExceptionType.Issued_expired_unit);


			// Now, test Caution Tag: 25
			ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(!ct.PrintOK && ct.Message == Common.StrRes.InfoMsg.UC043.UnitExpiredNoOverride().ResString, "CautionTag: Expired");

		}


		[Test]
		public void ValidateUnit_Fail_All_Others()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			BOL.BloodUnit bloodUnitWB = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E001", "E0001", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.WB);
			BOL.BloodUnit bloodUnitFFP = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E026", "E3465", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.FFP);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);

	            

			// Rh positive patient
			// setup failure data
			UnitTestUtil.RunSQL("UPDATE SpecimenTest SET " + TABLES.SpecimenTest.TestResultId + " = 'N' WHERE " + TABLES.SpecimenTest.SpecimenTestGuid + " IN ('" + _specimenTestRhTestOneGuid + "','" + _specimenTestRhTestTwoGuid + "')" );
		
			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);

			Hashtable results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));

			Assert.IsTrue(results.Count == 1);
			Assert.IsTrue(results.ContainsKey(Common.StrRes.ConfMsg.UC073.AboIncompatibleUnitIssued()));
			Assert.IsTrue((Common.ExceptionType) results[Common.StrRes.ConfMsg.UC073.AboIncompatibleUnitIssued()] == Common.ExceptionType.Unissuable);


			// Now, test Caution Tag: 26
			BOL.CautionTag ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(ct.PrintOK, "CautionTag: Frozen");

		
			// reset data
			UnitTestUtil.RunSQL("UPDATE SpecimenTest SET " + TABLES.SpecimenTest.TestResultId + " = 'P' WHERE SpecimenTestGuid IN ('" + _specimenTestRhTestOneGuid + "','" + _specimenTestRhTestTwoGuid + "')" );


			// setup test data
			Guid orderedUnitGuid = this.InsertOrderedUnit(bloodUnit.BloodUnitGuid);

			results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
		
			Assert.IsTrue(results.Count == 2 &&
				results.ContainsKey(Common.StrRes.InfoMsg.UC073.UnitIsAssigned())
				&& (Common.ExceptionType) results[Common.StrRes.ValidMsg.UC003.BR321()] == Common.ExceptionType.Antigen_positive_untested_units_issued);


			// Now, test Caution Tag: 27
			ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(ct.PrintOK, "CautionTag: OK");

		
			// reset data 
			UnitTestUtil.RunSQL("DELETE FROM OrderedUnit WHERE OrderedUnitGuid = '" + orderedUnitGuid + "'");



			// setup test data
			bloodUnit.BloodUnitStatus.BiohazardousWarningIndicator = true;

			results = _patientUnitIssue.ValidateUnit(bloodUnit, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
		
			Assert.IsTrue(results.Count == 2 &&
				results.ContainsKey(Common.StrRes.ConfMsg.UC073.Biohazardous())
				&&	(Common.ExceptionType) results[Common.StrRes.ConfMsg.UC073.Biohazardous()] == Common.ExceptionType.Biohazardous_unit_issued);


			// Now, test Caution Tag: 26: ToDo
			ct = new BOL.CautionTag(bloodUnit, new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist), false);
			Assert.IsTrue(ct.PrintOK, "CautionTag: Biohazard");


			// reset data
			bloodUnit.BloodUnitStatus.BiohazardousWarningIndicator = false;

			// System ABO for WB
			//results = _patientUnitIssue.ValidateUnit(bloodUnitWB, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
			// System ABO for FFP
			//results = _patientUnitIssue.ValidateUnit(bloodUnitFFP, new BOL.OrderedUnit(_orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false), true, new BOL.Specimen(_specimenTwoGuid));
		
		}


		
		[Test]
		public void ValidateWholeBloodTT_7301_Pass()
		{
			// Create unit for testing
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));

			string deleteString = InsertSystemAboInterps(Common.ABO.A);
			Assert.IsTrue(_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));
			UnitTestUtil.RunSQL(deleteString);

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			deleteString = InsertSystemAboInterps(Common.ABO.B);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));
			Assert.IsTrue(_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));
			UnitTestUtil.RunSQL(deleteString);

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			deleteString = InsertSystemAboInterps(Common.ABO.AB);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));
			Assert.IsTrue(_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));
			UnitTestUtil.RunSQL(deleteString);

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			deleteString = InsertSystemAboInterps(Common.ABO.O);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));
			Assert.IsTrue(_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));
			UnitTestUtil.RunSQL(deleteString);
		}
		[Test]
		public void ValidateWholeBloodTT_7301_Fail()
		{
			// Create unit for testing
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));

			#region system interp B

			// Insert non-matching ABO
			string deleteString = InsertSystemAboInterps(Common.ABO.B);

			// Unit: A
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));

			// Unit: AB
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));

			// Unit: O
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));

			UnitTestUtil.RunSQL(deleteString);

			#endregion


			#region system interp A

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.B;

			// Insert non-matching ABO
			deleteString = InsertSystemAboInterps(Common.ABO.A);

			// Unit: B
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));

			// Unit: AB
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));

			// Unit: O
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));

			UnitTestUtil.RunSQL(deleteString);

			#endregion


			#region system interp AB

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.A;

			// Insert non-matching ABO
			deleteString = InsertSystemAboInterps(Common.ABO.AB);

			// Unit: A
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));

			// Unit: B
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));

			// Unit: O
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));

			UnitTestUtil.RunSQL(deleteString);

			#endregion


			#region system interp O

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.A;

			// Insert non-matching ABO
			deleteString = InsertSystemAboInterps(Common.ABO.O);

			// Unit: A
			Assert.IsTrue(!_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));

			// Unit: B
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));

			// Unit: AB
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));

			UnitTestUtil.RunSQL(deleteString);

			#endregion


			#region system interp NA

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.A;

			// Insert non-matching ABO
			deleteString = InsertSystemAboInterps(Common.ABO.NA);

			// Unit: A
			Assert.IsTrue(!_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));

			// Unit: B
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			Assert.IsTrue(!_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));

			// Unit: AB
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			Assert.IsTrue(!_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));

			// Unit: O
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateWholeBloodTT_7301(bloodUnit, true));

			UnitTestUtil.RunSQL(deleteString);

			#endregion
		}


		[Test]
		public void ValidateNonWholeBloodNonPlasmaTT_7302_Pass()
		{
			// Create unit for testing
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));

			// system ABO: A
			string deleteString = InsertSystemAboInterps(Common.ABO.A);

			Assert.IsTrue(_patientUnitIssue.ValidateNonWholeBloodNonPlasmaTT_7302(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			Assert.IsTrue(_patientUnitIssue.ValidateNonWholeBloodNonPlasmaTT_7302(bloodUnit, true));
			UnitTestUtil.RunSQL(deleteString);


			// system ABO: B
			deleteString = InsertSystemAboInterps(Common.ABO.B);
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.B;

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));
			Assert.IsTrue(_patientUnitIssue.ValidateNonWholeBloodNonPlasmaTT_7302(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			Assert.IsTrue(_patientUnitIssue.ValidateNonWholeBloodNonPlasmaTT_7302(bloodUnit, true));
			UnitTestUtil.RunSQL(deleteString);


			// system ABO: AB
			deleteString = InsertSystemAboInterps(Common.ABO.AB);
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));
			Assert.IsTrue(_patientUnitIssue.ValidateNonWholeBloodNonPlasmaTT_7302(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			Assert.IsTrue(_patientUnitIssue.ValidateNonWholeBloodNonPlasmaTT_7302(bloodUnit, true));
			UnitTestUtil.RunSQL(deleteString);


			// system ABO: O
			deleteString = InsertSystemAboInterps(Common.ABO.O);
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.O;

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));
			Assert.IsTrue(_patientUnitIssue.ValidateNonWholeBloodNonPlasmaTT_7302(bloodUnit, true));
			UnitTestUtil.RunSQL(deleteString);


			// system ABO: ???
			deleteString = InsertSystemAboInterps(Common.ABO.NA);
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.O;

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.LeadTechnologist));
			Assert.IsTrue(_patientUnitIssue.ValidateNonWholeBloodNonPlasmaTT_7302(bloodUnit, true));
			UnitTestUtil.RunSQL(deleteString);
		}
		[Test]
		public void ValidateNonWholeBloodNonPlasmaTT_7302_Fail()
		{
			// Create unit for testing
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			
			// system ABO: B
			string deleteString = InsertSystemAboInterps(Common.ABO.B);

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateNonWholeBloodNonPlasmaTT_7302(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateNonWholeBloodNonPlasmaTT_7302(bloodUnit, true));
			UnitTestUtil.RunSQL(deleteString);


			// system ABO: A
			deleteString = InsertSystemAboInterps(Common.ABO.A);
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.B;

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateNonWholeBloodNonPlasmaTT_7302(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			Assert.IsTrue(!_patientUnitIssue.ValidateNonWholeBloodNonPlasmaTT_7302(bloodUnit, true));
			UnitTestUtil.RunSQL(deleteString);


			// system ABO: AB
			deleteString = InsertSystemAboInterps(Common.ABO.AB);
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateNonWholeBloodNonPlasmaTT_7302(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateNonWholeBloodNonPlasmaTT_7302(bloodUnit, true));
			UnitTestUtil.RunSQL(deleteString);


			// system ABO: O
			deleteString = InsertSystemAboInterps(Common.ABO.O);
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.A;

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateNonWholeBloodNonPlasmaTT_7302(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateNonWholeBloodNonPlasmaTT_7302(bloodUnit, true));
	
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.ValidateNonWholeBloodNonPlasmaTT_7302(bloodUnit, true));

			UnitTestUtil.RunSQL(deleteString);
		}


		[Test]
		public void ValidatePlasmaTT73_03_Pass()
		{
			// Create unit for testing
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E014", "E1550", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.FFP);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));

			// system ABO: A
			string deleteString = InsertSystemAboInterps(Common.ABO.A);

			Assert.IsTrue(_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));
			UnitTestUtil.RunSQL(deleteString);


			// system ABO: B
			deleteString = InsertSystemAboInterps(Common.ABO.B);
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.B;

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));
			UnitTestUtil.RunSQL(deleteString);


			// system ABO: AB
			deleteString = InsertSystemAboInterps(Common.ABO.AB);
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));
			UnitTestUtil.RunSQL(deleteString);


			// system ABO: O
			deleteString = InsertSystemAboInterps(Common.ABO.O);
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));


			// system ABO: ????
			deleteString = InsertSystemAboInterps(Common.ABO.NA);
			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));

			UnitTestUtil.RunSQL(deleteString);


		}
		[Test]
		public void ValidatePlasmaTT73_03_Fail()
		{
			// Create unit for testing
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));

			// system ABO: B
			string deleteString = InsertSystemAboInterps(Common.ABO.B);

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));
			UnitTestUtil.RunSQL(deleteString);


			// system ABO: A
			deleteString = InsertSystemAboInterps(Common.ABO.A);

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			Assert.IsTrue(!_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));
			UnitTestUtil.RunSQL(deleteString);


			// system ABO: AB
			deleteString = InsertSystemAboInterps(Common.ABO.AB);

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));

			UnitTestUtil.RunSQL(deleteString);


			// system ABO: NA
			deleteString = InsertSystemAboInterps(Common.ABO.NA);

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));

			bloodUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.ValidatePlasmaTT73_03(bloodUnit, true));

			UnitTestUtil.RunSQL(deleteString);
		}

		[Test]
		public void InitializeReleaseDataTable_Pass()
		{
			DataTable dtRel=BOL.PatientUnitIssue.InitializeReleaseDataTable();

			Assert.IsNotNull(dtRel);
			// ToDo: replace with column count
			Assert.IsTrue(dtRel.Columns.Count>0);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void InitializeReleaseDataTable_Fail()
		{
		}


		[Test]
		public void Beep_Pass()
		{
			bool UC073_UnitFrozen=BOL.PatientUnitIssue.Beep( Common.StrRes.InfoMsg.UC073.UnitFrozen() );
			bool UC073_SpecimenExpired=BOL.PatientUnitIssue.Beep( Common.StrRes.ConfMsg.UC073.SpecimenExpired() );
			bool UC073_UnitExpired=BOL.PatientUnitIssue.Beep( Common.StrRes.ConfMsg.UC073.UnitExpired() );
			bool UC073_AntigenAntibodyProblem=BOL.PatientUnitIssue.Beep( Common.StrRes.ConfMsg.UC073.AntibodyProblem(string.Empty, string.Empty, string.Empty) );
			bool UC073_RestrictiveUnits=BOL.PatientUnitIssue.Beep( Common.StrRes.ConfMsg.UC073.RestrictiveUnits() );
			bool UC073_TRNotSatisfied=BOL.PatientUnitIssue.Beep( Common.StrRes.ConfMsg.UC073.TRNotSatisfied(string.Empty) );
			bool UC073_Biohazardous=BOL.PatientUnitIssue.Beep( Common.StrRes.ConfMsg.UC073.Biohazardous() );
			bool UC073_EmergencyIssue=BOL.PatientUnitIssue.Beep( Common.StrRes.ConfMsg.UC073.EmergencyIssue() );
			bool UC073_AboIncompatibleUnitIssued=BOL.PatientUnitIssue.Beep( Common.StrRes.ConfMsg.UC073.AboIncompatibleUnitIssued() );
			bool UC003_BR321=BOL.PatientUnitIssue.Beep( Common.StrRes.ValidMsg.UC003.BR321() );
			
			Assert.IsTrue(UC073_UnitFrozen);
			Assert.IsTrue(UC073_SpecimenExpired);
			Assert.IsTrue(UC073_UnitExpired);
			Assert.IsTrue(UC073_AntigenAntibodyProblem);
			Assert.IsTrue(UC073_RestrictiveUnits);
			Assert.IsTrue(UC073_TRNotSatisfied);
			Assert.IsTrue(UC073_Biohazardous);
			Assert.IsTrue(UC073_EmergencyIssue);
			Assert.IsTrue(UC073_AboIncompatibleUnitIssued);
			Assert.IsTrue(UC003_BR321);
		}
		/*[Test]
		public void Beep_Fail()
		{
			bool fail=BOL.PatientUnitIssue.Beep( Common.StrRes.ConfMsg.UC001.BR1_40() );
			
			Assertion.IsTrue(!fail);
		}*/

		[Test]
		public void GetSystemAbo_Pass()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));

			string deleteString = this.InsertSystemAboInterps(Common.ABO.A);
			Assert.IsTrue(_patientUnitIssue.GetSystemAbo() == "A");
			UnitTestUtil.RunSQL(deleteString);

			deleteString = this.InsertSystemAboInterps(Common.ABO.B);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(_patientUnitIssue.GetSystemAbo() == "B");
			UnitTestUtil.RunSQL(deleteString);

			deleteString = this.InsertSystemAboInterps(Common.ABO.AB);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(_patientUnitIssue.GetSystemAbo() == "AB");
			UnitTestUtil.RunSQL(deleteString);

			deleteString = this.InsertSystemAboInterps(Common.ABO.O);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(_patientUnitIssue.GetSystemAbo() == "O");
			UnitTestUtil.RunSQL(deleteString);
		}
		[Test]
		public void GetSystemAbo_Fail()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);

			string deleteString = this.InsertSystemAboInterps(Common.ABO.I);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(_patientUnitIssue.GetSystemAbo() == "I");
			UnitTestUtil.RunSQL(deleteString);
		}


		[Test]
		public void GetSystemRh_Pass()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			
			string deleteString = this.InsertSystemRhInterps(Common.RH.Positive);
			Assert.IsTrue(_patientUnitIssue.GetSystemRh() == "P");
			UnitTestUtil.RunSQL(deleteString);

			deleteString = this.InsertSystemRhInterps(Common.RH.Negative);
			Assert.IsTrue(_patientUnitIssue.GetSystemRh() == "N");
			UnitTestUtil.RunSQL(deleteString);
		}
		[Test]
		public void GetSystemRh_Fail()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), this.GetVbecsUser(Common.UserRoleType.Technologist));
			
			string deleteString = this.InsertSystemRhInterps(Common.RH.Inconclusive);
			Assert.IsTrue(_patientUnitIssue.GetSystemRh() == "I");
			UnitTestUtil.RunSQL(deleteString);
		}

		#endregion

		
		#region inner class		

		#region constructors

		[Test]
		public void PatientUnitIssue_IssuedUnits_Constructor_Pass()
		{
			BOL.PatientUnitIssue.IssuedUnits issuedUnits = new gov.va.med.vbecs.BOL.PatientUnitIssue.IssuedUnits(DateTime.Today);
			Assert.IsNotNull(issuedUnits);
			Assert.IsTrue(issuedUnits.Count==0);
		}
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void PatientUnitIssue_IssuedUnits_Constructor_Fail()
		{
			BOL.PatientUnitIssue.IssuedUnits issuedUnits = new gov.va.med.vbecs.BOL.PatientUnitIssue.IssuedUnits(DateTime.MinValue);
			
		}


		[Test]
		public void PatientUnitIssue_IssuedUnits_AddUnit_Pass()
		{
			BOL.PatientUnitIssue.IssuedUnits issuedUnits = new gov.va.med.vbecs.BOL.PatientUnitIssue.IssuedUnits(DateTime.Today);
			Assert.IsNotNull(issuedUnits);
			Assert.IsTrue(issuedUnits.Count==0);
		}
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void PatientUnitIssue_IssuedUnits_AddUnit_Fail()
		{
			BOL.PatientUnitIssue.IssuedUnits issuedUnits = new gov.va.med.vbecs.BOL.PatientUnitIssue.IssuedUnits(DateTime.MinValue);
			
		}

		#endregion


		[Test]
		public void IssuedUnits_SaveUnits_Pass()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));

			_patientUnitIssue.AddUnit(bloodUnit, false);

			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.IssueUnit);
			bool success = _patientUnitIssue.Save(Common.WorkloadProcessID.IssueUnit);
			Assert.IsTrue(success);	

			
			// Emergency issue
			bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));

			_patientUnitIssue.AddUnit(bloodUnit, true);

			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.IssueUnit);
			success = _patientUnitIssue.Save(Common.WorkloadProcessID.IssueUnit);
			Assert.IsTrue(success);	
		}
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void IssuedUnits_SaveUnits_Fail()
		{
			DataTable dtOne = null;
			DataTable dtTwo = null;
			_issuedUnits.SaveUnits(Common.UpdateFunction.Invalid, dtOne, dtTwo, Common.WorkloadProcessID.Unknown, new ArrayList());		
		}

		#endregion

		#endregion

		/// <summary>
		/// Select Autologous
		/// </summary>
		[Test]
		public void SelectionOrderValid_Pass()
		{
			// Setup
			BOL.BloodUnit bloodUnitAutologous = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			BOL.BloodUnit bloodUnitDirected = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			Guid [] bloodUnitGuids = {bloodUnitAutologous.BloodUnitGuid, bloodUnitDirected.BloodUnitGuid};  


			#region Make units directed/ autologous

			// AutologousPut in transaction
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");
			string donate = "UPDATE BloodUnit SET DonationTypeId = " + (int) Common.DonationType.ForAutologousUseOnly + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'";
			UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 10 + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");

			// Directed
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");
			UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 14 + " WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");

			// Regenerate BUs so they pick up the update donation codes
			bloodUnitAutologous = new BOL.BloodUnit(bloodUnitAutologous.BloodUnitGuid);
			bloodUnitDirected = new BOL.BloodUnit(bloodUnitDirected.BloodUnitGuid);

			#endregion

			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnitGuids);

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(_patientUnitIssue.SelectionOrderValid(bloodUnitAutologous));
		}
		/// <summary>
		/// Select directed after autologous
		/// </summary>
		[Test]
		public void SelectionOrderValid_Two_Pass()
		{
			// Setup
			BOL.BloodUnit bloodUnitAutologous = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			BOL.BloodUnit bloodUnitDirected = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			Guid [] bloodUnitGuids = {bloodUnitAutologous.BloodUnitGuid, bloodUnitDirected.BloodUnitGuid};  


			#region Make units directed/ autologous

			// AutologousPut in transaction
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");
			string donate = "UPDATE BloodUnit SET DonationTypeId = " + (int) Common.DonationType.ForAutologousUseOnly + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'";
			UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 10 + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");

			// Directed
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");
			UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 14 + " WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");

			// Regenerate BUs so they pick up the update donation codes
			bloodUnitAutologous = new BOL.BloodUnit(bloodUnitAutologous.BloodUnitGuid);
			bloodUnitDirected = new BOL.BloodUnit(bloodUnitDirected.BloodUnitGuid);

			#endregion

			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnitGuids);

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			_patientUnitIssue.AddUnit(bloodUnitAutologous, false);
			Assert.IsTrue(_patientUnitIssue.SelectionOrderValid(bloodUnitDirected));
		}
		[Test]
		public void SelectionOrderValid_Three_Pass()
		{
			// Setup
			BOL.BloodUnit bloodUnitAutologous = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			BOL.BloodUnit bloodUnitDirected = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			BOL.BloodUnit bloodUnitRestricted = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			Guid [] bloodUnitGuids = {bloodUnitAutologous.BloodUnitGuid, bloodUnitDirected.BloodUnitGuid, bloodUnitRestricted.BloodUnitGuid};  


			#region Make units directed/ autologous

			// AutologousPut in transaction
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");
			string donate = "UPDATE BloodUnit SET DonationTypeId = " + (int) Common.DonationType.ForAutologousUseOnly + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'";
			UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 10 + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");

			// Directed
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");
			UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 14 + " WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");

			// Restricted
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");
			
			// Regenerate BUs so they pick up the update donation codes
			bloodUnitAutologous = new BOL.BloodUnit(bloodUnitAutologous.BloodUnitGuid);
			bloodUnitDirected = new BOL.BloodUnit(bloodUnitDirected.BloodUnitGuid);
			bloodUnitRestricted = new BOL.BloodUnit(bloodUnitRestricted.BloodUnitGuid);

			#endregion

			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnitGuids);

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			_patientUnitIssue.AddUnit(bloodUnitAutologous, false);
			_patientUnitIssue.AddUnit(bloodUnitDirected, false);
			Assert.IsTrue(_patientUnitIssue.SelectionOrderValid(bloodUnitRestricted));
		}
		[Test]
		public void SelectionOrderValid_Four_Pass()
		{
			// Setup
			BOL.BloodUnit bloodUnitAutologous = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			BOL.BloodUnit bloodUnitDirected = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			BOL.BloodUnit bloodUnitRestricted = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			BOL.BloodUnit bloodUnitAllogeneic = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			Guid [] bloodUnitGuids = {bloodUnitAutologous.BloodUnitGuid, bloodUnitDirected.BloodUnitGuid, bloodUnitRestricted.BloodUnitGuid, bloodUnitAllogeneic.BloodUnitGuid};  


			#region Make units directed/ autologous

			// AutologousPut in transaction
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");
			string donate = "UPDATE BloodUnit SET DonationTypeId = " + (int) Common.DonationType.ForAutologousUseOnly + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'";
			UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 10 + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");

			// Directed
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");
			UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 14 + " WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");

			// Restricted
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitRestricted.BloodUnitGuid + "'");

			// Regenerate BUs so they pick up the update donation codes
			bloodUnitAutologous = new BOL.BloodUnit(bloodUnitAutologous.BloodUnitGuid);
			bloodUnitDirected = new BOL.BloodUnit(bloodUnitDirected.BloodUnitGuid);
			bloodUnitRestricted = new BOL.BloodUnit(bloodUnitRestricted.BloodUnitGuid);

			#endregion

			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnitGuids);

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			_patientUnitIssue.AddUnit(bloodUnitAutologous, false);
			_patientUnitIssue.AddUnit(bloodUnitDirected, false);
			_patientUnitIssue.AddUnit(bloodUnitRestricted, false);
			Assert.IsTrue(_patientUnitIssue.SelectionOrderValid(bloodUnitAllogeneic));
		}


		/// <summary>
		/// Select directed before autologous
		/// </summary>
		[Test]
		public void SelectionOrderValid_Fail()
		{
			// Setup
			BOL.BloodUnit bloodUnitAutologous = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			BOL.BloodUnit bloodUnitDirected = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			Guid [] bloodUnitGuids = {bloodUnitAutologous.BloodUnitGuid, bloodUnitDirected.BloodUnitGuid};  


			#region Make units directed/ autologous

			// AutologousPut in transaction
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");
			string donate = "UPDATE BloodUnit SET DonationTypeId = " + (int) Common.DonationType.ForAutologousUseOnly + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'";
			UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 10 + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");

			// Directed
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");
			UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 14 + " WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");

			// Regenerate BUs so they pick up the update donation codes
			bloodUnitAutologous = new BOL.BloodUnit(bloodUnitAutologous.BloodUnitGuid);
			bloodUnitDirected = new BOL.BloodUnit(bloodUnitDirected.BloodUnitGuid);

			#endregion

			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnitGuids);

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.SelectionOrderValid(bloodUnitDirected));
		}


		/// <summary>
		/// Select allogeneic before autologous
		/// </summary>
		[Test]
		public void SelectionOrderValid_Two_Fail()
		{
			// Setup
			BOL.BloodUnit bloodUnitAutologous = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			BOL.BloodUnit bloodUnitAllogeneic = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			Guid [] bloodUnitGuids = {bloodUnitAutologous.BloodUnitGuid, bloodUnitAllogeneic.BloodUnitGuid};  


			#region Make units directed/ autologous

			// AutologousPut in transaction
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");
			string donate = "UPDATE BloodUnit SET DonationTypeId = " + (int) Common.DonationType.ForAutologousUseOnly + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'";
			UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 10 + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");

			// Regenerate BUs so they pick up the update donation codes
			bloodUnitAutologous = new BOL.BloodUnit(bloodUnitAutologous.BloodUnitGuid);
			bloodUnitAllogeneic = new BOL.BloodUnit(bloodUnitAllogeneic.BloodUnitGuid);

			#endregion

			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnitGuids);

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.SelectionOrderValid(bloodUnitAllogeneic));
		}
		/// <summary>
		/// Select allogeneic before directed
		/// </summary>
		[Test]
		public void SelectionOrderValid_Three_Fail()
		{
			// Setup
			BOL.BloodUnit bloodUnitDirected = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			BOL.BloodUnit bloodUnitAllogeneic = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			Guid [] bloodUnitGuids = {bloodUnitDirected.BloodUnitGuid, bloodUnitAllogeneic.BloodUnitGuid};  


			#region Make units directed

			// Directed
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");
			UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 14 + " WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");

			// Regenerate BUs so they pick up the update donation codes
			bloodUnitDirected = new BOL.BloodUnit(bloodUnitDirected.BloodUnitGuid);
			bloodUnitAllogeneic = new BOL.BloodUnit(bloodUnitAllogeneic.BloodUnitGuid);

			#endregion

			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnitGuids);

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.SelectionOrderValid(bloodUnitAllogeneic));
		}
		/// <summary>
		/// Select allogeneic before restricted
		/// </summary>
		[Test]
		public void SelectionOrderValid_Four_Fail()
		{
			// Setup
			BOL.BloodUnit bloodUnitRestricted = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			BOL.BloodUnit bloodUnitAllogeneic = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			Guid [] bloodUnitGuids = {bloodUnitRestricted.BloodUnitGuid, bloodUnitAllogeneic.BloodUnitGuid};  


			#region Make unit restricted

			// Restricted
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitRestricted.BloodUnitGuid + "'");
			//UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 14 + " WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");

			// Regenerate BUs so they pick up the update donation codes
			bloodUnitRestricted = new BOL.BloodUnit(bloodUnitRestricted.BloodUnitGuid);
			bloodUnitAllogeneic = new BOL.BloodUnit(bloodUnitAllogeneic.BloodUnitGuid);

			#endregion

			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnitGuids);

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.SelectionOrderValid(bloodUnitAllogeneic));
		}
		/// <summary>
		/// Select restricted before autologous
		/// </summary>
		[Test]
		public void SelectionOrderValid_Five_Fail()
		{
			// Setup
			BOL.BloodUnit bloodUnitAutologous = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			BOL.BloodUnit bloodUnitRestricted = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			Guid [] bloodUnitGuids = {bloodUnitAutologous.BloodUnitGuid, bloodUnitRestricted.BloodUnitGuid};  


			#region Make units restricted/ autologous

			// AutologousPut in transaction
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");
			string donate = "UPDATE BloodUnit SET DonationTypeId = " + (int) Common.DonationType.ForAutologousUseOnly + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'";
			UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 10 + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");

			// Make Restricted
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitRestricted.BloodUnitGuid + "'");

			// Regenerate BUs so they pick up the update donation codes
			bloodUnitAutologous = new BOL.BloodUnit(bloodUnitAutologous.BloodUnitGuid);
			bloodUnitRestricted = new BOL.BloodUnit(bloodUnitRestricted.BloodUnitGuid);

			#endregion

			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnitGuids);

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.SelectionOrderValid(bloodUnitRestricted));
		}
		/// <summary>
		/// Select restricted before directed
		/// </summary>
		[Test]
		public void SelectionOrderValid_Six_Fail()
		{
			// Setup
			BOL.BloodUnit bloodUnitDirected = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			BOL.BloodUnit bloodUnitRestricted = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			Guid [] bloodUnitGuids = {bloodUnitDirected.BloodUnitGuid, bloodUnitRestricted.BloodUnitGuid};  


			#region Make units directed/ autologous

			// Directed
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");
			UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 14 + " WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");

			// Regenerate BUs so they pick up the update donation codes
			bloodUnitDirected = new BOL.BloodUnit(bloodUnitDirected.BloodUnitGuid);
			bloodUnitRestricted = new BOL.BloodUnit(bloodUnitRestricted.BloodUnitGuid);

			#endregion

			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnitGuids);

			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(!_patientUnitIssue.SelectionOrderValid(bloodUnitRestricted));
		}


		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LoadDataRowFromThis_Pass()
		{
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LoadDataRowFromThis_Fail()
		{
		}


		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssuedUnits_GetIssuedUnits_Pass()
		{
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssuedUnits_GetIssuedUnits_Fail()
		{
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssuedUnits_RemoveUnit_Pass()
		{
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssuedUnits_RemoveUnit_Fail()
		{
		}


		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void IssuedUnits_GetIssuedUnit_Pass()
		{
			// Setup
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);
			_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			_patientUnitIssue.AddUnit(bloodUnit, false);
			_issuedUnits.GetIssuedUnit(bloodUnit.BloodUnitGuid);
			Assert.IsTrue(true);
		}
		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void IssuedUnits_GetIssuedUnit_Fail()
		{
			_issuedUnits.GetIssuedUnit(Guid.NewGuid());
		}


		// Begin IEnumerable

		[Test]
		public void GetEnumerator_Pass()
		{
			_issuedUnits.GetEnumerator();
			Assert.IsTrue(_issuedUnits.GetEnumerator() != null);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetEnumerator_Fail()
		{
		}

		// End IEnumerable


		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssuedUnitEnumerator_MoveNext_Pass()
		{
		
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssuedUnitEnumerator_MoveNext_Fail()
		{
		}


		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssuedUnitEnumerator_Reset_Pass()
		{
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssuedUnitEnumerator_Reset_Fail()
		{
		}


		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssuedUnitEnumerator_Current_Pass()
		{
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IssuedUnitEnumerator_Current_Fail()
		{
		}

		
		[Test]
		public void UnitsAvailableForLevelThree_Pass()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			this.CreateTestData(_patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true, true, "C", false, bloodUnit.BloodUnitGuid);

			// throw in an override just to make sure this is working
			this.SetupAllCRs(new BOL.Patient(_patientGuid));

			System.Collections.ArrayList bloodUnitGuids = new ArrayList();
			bloodUnitGuids.Add(bloodUnit.BloodUnitGuid);

			Assert.IsTrue( BOL.PatientUnitIssue.UnitsAvailableForLevelThree(new BOL.Patient(_patientGuid), bloodUnitGuids) > 0 );
			DeletePatientData(_patientGuid);

			// Reset
			UnitTestUtil.RunSQL("DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + _patientGuid + "'");

		}
		[Test]
		public void UnitsAvailableForLevelThree_Fail()
		{
			System.Collections.ArrayList bloodUnitGuids = new ArrayList();
			bloodUnitGuids.Add(System.Guid.NewGuid());

			Assert.IsTrue(BOL.PatientUnitIssue.UnitsAvailableForLevelThree(new BOL.Patient(_patientGuid), bloodUnitGuids) == 0 );
		}



		#region helper methods

		private Guid SetupSpecimenTest(Guid specimenGuid, Guid orderedTestGuid, string test, string result)
		{
			StringBuilder query = new StringBuilder();
			Guid specimenTestGuid = Guid.NewGuid();

			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestGuid).Append("',");
			query.Append("'").Append(specimenGuid).Append("',");
			query.Append("'").Append(orderedTestGuid).Append("',");
			query.Append("'" + test + "',");
			query.Append("'"+ result +"',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			UnitTestUtil.RunSQL(query.ToString());

			return specimenTestGuid;
		}


		private string InsertSystemAboInterps(Common.ABO abo)
		{
			string deleteABOs = "DELETE FROM SpecimenTest WHERE " + TABLES.SpecimenTest.BloodTestTypeId + " IN(" + (int) Common.TestType.AntiA_TAS + "," +  (int) Common.TestType.AntiB_TAS + "," + (int) Common.TestType.AntiAB_TAS + "," + (int) Common.TestType.A1Cells + "," + (int) Common.TestType.BCells + ")";
			
			UnitTestUtil.RunSQL(deleteABOs);

			StringBuilder query = new StringBuilder();
			Guid specimenTestAntiAGuid		= Guid.NewGuid();
			Guid specimenTestAntiBGuid		= Guid.NewGuid();
			Guid specimenTestAntiABGuid		= Guid.NewGuid();
			Guid specimenTestA1CellsGuid	= Guid.NewGuid();
			Guid specimenTestBCellsGuid		= Guid.NewGuid();

			string antiA_TAS	= string.Empty;
			string antiB_TAS	= string.Empty;
			string antiAB_TAS	= string.Empty;
			string A1Cells		= string.Empty;
			string BCells		= string.Empty;

			switch (abo)
			{
				case Common.ABO.A:
					antiA_TAS	= "P";
					antiB_TAS	= "N";
					antiAB_TAS	= "X";
					A1Cells		= "N";	// doesn't matter
					BCells		= "P";	// doesn't matter
					break;
				case Common.ABO.B:
					antiA_TAS	= "N";
					antiB_TAS	= "P";
					antiAB_TAS	= "X";
					A1Cells		= "P";	// doesn't matter
					BCells		= "N";	// doesn't matter
					break;
				case Common.ABO.AB:
					antiA_TAS	= "P";
					antiB_TAS	= "P";
					antiAB_TAS	= "X";
					A1Cells		= "N";	// doesn't matter
					BCells		= "N";	// doesn't matter
					break;
				case Common.ABO.O:
					antiA_TAS	= "N";
					antiB_TAS	= "N";
					antiAB_TAS	= "X";
					A1Cells		= "P";	// doesn't matter
					BCells		= "P";	// doesn't mattery;
					break;
				case Common.ABO.I:
					antiA_TAS	= "N";
					antiB_TAS	= "N";
					antiAB_TAS	= "N";
					A1Cells		= "N";
					BCells		= "N";
					break;
				default:
					antiA_TAS	= "N";
					antiB_TAS	= "N";
					antiAB_TAS	= "N";
					A1Cells		= "N";
					BCells		= "N";
					break;
			}

			#region insert

			query.Append("BEGIN TRANSACTION T1 ");
			// Specimen (#2) test: (system generated Anti-A TAS)
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAntiAGuid.ToString()).Append("',");
			query.Append("'").Append(_specimenTwoGuid.ToString()).Append("',");
			query.Append("'").Append(_orderedTestGuid.ToString()).Append("',");
			query.Append( (int) Common.TestType.AntiA_TAS + ",");
			query.Append("'" + antiA_TAS+ "',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// Specimen (#2) test: (system generated Anti-B TAS)
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAntiBGuid.ToString()).Append("',");
			query.Append("'").Append(_specimenTwoGuid.ToString()).Append("',");
			query.Append("'").Append(_orderedTestGuid.ToString()).Append("',");
			query.Append( (int) Common.TestType.AntiB_TAS + ",");
			query.Append("'" + antiB_TAS + "',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// Specimen (#2) test: (system generated Anti-AB TAS)
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAntiABGuid.ToString()).Append("',");
			query.Append("'").Append(_specimenTwoGuid.ToString()).Append("',");
			query.Append("'").Append(_orderedTestGuid.ToString()).Append("',");
			query.Append( (int) Common.TestType.AntiAB_TAS + ",");
			query.Append("'" + antiAB_TAS + "',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// Specimen (#2) test: (system generated A1Cells TAS)
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestA1CellsGuid.ToString()).Append("',");
			query.Append("'").Append(_specimenTwoGuid.ToString()).Append("',");
			query.Append("'").Append(_orderedTestGuid.ToString()).Append("',");
			query.Append( (int) Common.TestType.A1Cells + ",");
			query.Append("'" + A1Cells + "',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// Specimen (#2) test: (system generated BCells TAS)
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestBCellsGuid.ToString()).Append("',");
			query.Append("'").Append(_specimenTwoGuid.ToString()).Append("',");
			query.Append("'").Append(_orderedTestGuid.ToString()).Append("',");
			query.Append( (int) Common.TestType.BCells + ",");
			query.Append("'" + BCells + "',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			query.Append("COMMIT TRANSACTION T1 ");

			UnitTestUtil.RunSQL(query.ToString());

			#endregion
			string delete = "DELETE FROM SpecimenTest WHERE SpecimenTestGuid IN ('" + specimenTestAntiAGuid + "','" + specimenTestAntiBGuid + "','" + specimenTestAntiABGuid +"','" + specimenTestA1CellsGuid + "','" + specimenTestBCellsGuid + "')";
			return delete;
		}


		private string InsertSystemRhInterps(Common.RH rH)
		{
			string deleteABOs = "DELETE FROM SpecimenTest WHERE " + TABLES.SpecimenTest.BloodTestTypeId + " IN (" + (int) Common.TestType.AntiD_TAS + "," +  (int) Common.TestType.DControl + ")";
			
			UnitTestUtil.RunSQL(deleteABOs);

			StringBuilder query = new StringBuilder();
			Guid specimenTestAntiDTASGuid	= Guid.NewGuid();
			Guid specimenTestDControlGuid	= Guid.NewGuid();

			string antiD_TAS	= string.Empty;
			string DControl		= string.Empty;
			
			switch (rH)
			{
				case Common.RH.Positive:
					antiD_TAS	= "P";
					DControl	= "N";
					break;
				case Common.RH.Negative:
					antiD_TAS	= "N";
					DControl	= "N";
					break;
				default:
					antiD_TAS	= "X";
					DControl	= "X";
					break;
			}

			#region insert

			query.Append("BEGIN TRANSACTION T1 ");
			// Specimen (#2) test: (system generated Anti-D TAS)
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAntiDTASGuid.ToString()).Append("',");
			query.Append("'").Append(_specimenTwoGuid.ToString()).Append("',");
			query.Append("'").Append(_orderedTestGuid.ToString()).Append("',");
			query.Append( (int) Common.TestType.AntiD_TAS + ",");
			query.Append("'" + antiD_TAS+ "',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// Specimen (#2) test: (system generated DControl)
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestDControlGuid.ToString()).Append("',");
			query.Append("'").Append(_specimenTwoGuid.ToString()).Append("',");
			query.Append("'").Append(_orderedTestGuid.ToString()).Append("',");
			query.Append( (int) Common.TestType.DControl + ",");
			query.Append("'" + DControl + "',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			query.Append("COMMIT TRANSACTION T1 ");

			UnitTestUtil.RunSQL(query.ToString());

			#endregion

			string deleteString = "DELETE FROM SpecimenTest WHERE SpecimenTestGuid IN ('" + specimenTestAntiDTASGuid + "','" + specimenTestDControlGuid + "')"; 
			
			return deleteString;
		}


		private void DeletePatientData(Guid patientGuid)
		{
			string delete = "BEGIN TRANSACTION DeleteItAll ";

			delete += "DELETE FROM BloodUnitTest WHERE OrderedUnitGuid IN (SELECT OrderedUnitGuid FROM OrderedUnit ";
			delete += "JOIN OrderedComponent OC ON OC.OrderedComponentGUID = OrderedUnit.OrderedComponentGUID ";
			delete += "JOIN PatientOrder PO ON OC.PatientOrderGUID = PO.PatientOrderGUID ";
			delete += "JOIN PatientTreatment PT ON PT.PatientTreatmentGUID = PO.PatientTreatmentGUID ";
			delete += "WHERE PT.PatientGuid = '" + patientGuid + "')";

			delete += "\n";

			delete += "DELETE FROM OrderedUnit WHERE OrderedUnitGuid IN (SELECT OrderedUnitGuid FROM OrderedUnit ";
			delete += "JOIN OrderedComponent OC ON OC.OrderedComponentGUID = OrderedUnit.OrderedComponentGUID ";
			delete += "JOIN PatientOrder PO ON OC.PatientOrderGUID = PO.PatientOrderGUID ";
			delete += "JOIN PatientTreatment PT ON PT.PatientTreatmentGUID = PO.PatientTreatmentGUID ";
			delete += "WHERE PT.PatientGuid = '" + patientGuid + "')";

			delete += "COMMIT TRANSACTION DeleteItAll";
			//
			UnitTestUtil.RunSQL(delete);
		}


		private Guid InsertOrderedUnit(Guid bloodUnitGuid)
		{
			Guid orderedUnitGuid = Guid.NewGuid();
			
			StringBuilder query = new StringBuilder();
			// Ordered Unit
			query.Append("INSERT INTO ORDEREDUNIT (");
			query.Append(TABLES.OrderedUnit.OrderedUnitGuid).Append(",");
			query.Append(TABLES.OrderedUnit.OrderedComponentGuid).Append(",");
			query.Append(TABLES.OrderedUnit.BloodUnitGuid).Append(",");
			query.Append(TABLES.OrderedUnit.CrossmatchDate).Append(",");
			query.Append(TABLES.OrderedUnit.CrossmatchRequiredIndicator).Append(",");
			query.Append(TABLES.OrderedUnit.CrossmatchResultCode).Append(",");
			query.Append(TABLES.OrderedUnit.CrossmatchTechId).Append(",");
			query.Append(TABLES.OrderedUnit.RecordStatusCode).Append(",");
			query.Append(TABLES.OrderedUnit.DivisionCode).Append(",");
			query.Append(TABLES.OrderedUnit.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedUnit.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedUnit.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'" + orderedUnitGuid.ToString() + "',");
			query.Append("'" + _orderedComponentGuid + "',");
			query.Append("'" + bloodUnitGuid + "',");
			query.Append("'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',");
			query.Append("1,");
			query.Append("'C',");
			query.Append("'" + Environment.UserName + "',");
			query.Append("'A',");
			query.Append("'" + Common.LogonUser.LogonUserDivisionCode + "',");
			query.Append("'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',");
			query.Append("'" + Environment.UserName + "',");
			query.Append("'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')");
			query.Append("\n");
			UnitTestUtil.RunSQL(query.ToString());

			return orderedUnitGuid;
		}
        // The sets up the CR requirements on the patient.
        private void SetupCRs(BOL.Patient patient, bool irradiated, bool leukoreduced, bool washedRBC, bool washedPLT, bool CMV, bool SC)
        {
            DataTable dtInsertSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(false);
            DataTable dtUpdateSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(true);
            DataTable dtInsertTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(false);
            DataTable dtUpdateTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(true);

            BOL.PatientTransfusionRequirement pTR;
            Common.ComponentRequirement req;

            // clear it out
            UnitTestUtil.RunSQL("DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + patient.PatientGuid + "'");

            // Irradiated
            if (irradiated)
            {
                pTR = new BOL.PatientTransfusionRequirement(patient, "Irradiated cellular products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
                req = Common.ComponentRequirement.IRRADIATE;
                AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);
            }


            // Leuko
            if (leukoreduced)
            {
                pTR = new BOL.PatientTransfusionRequirement(patient, "Leuko-reduced cellular products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
                req = Common.ComponentRequirement.LEUKORED;
                AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);
            }


            // Washed RBC
            if (washedRBC)
            {
                pTR = new BOL.PatientTransfusionRequirement(patient, "Washed RBC products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
                req = Common.ComponentRequirement.WASHEDRBC;
                AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);
            }


            // Washed PLT
            if (washedPLT)
            {
                pTR = new BOL.PatientTransfusionRequirement(patient, "Washed PLT products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
                req = Common.ComponentRequirement.WASHEDPLT;
                AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);
            }


            // CMV Neg
            if (CMV)
            {
                pTR = new BOL.PatientTransfusionRequirement(patient, "CMV Negative cellular products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
                req = Common.ComponentRequirement.CMVNEG;
                AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);
            }


            // Sickle cell
            if (SC)
            {
                pTR = new BOL.PatientTransfusionRequirement(patient, "Sickle Cell Negative RBC products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
                req = Common.ComponentRequirement.SCNEG;
                AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);
            }

            bool success = BOL.Patient.SaveSIsAndTRs(dtInsertSpecialInstructions, dtUpdateSpecialInstructions, dtInsertTransfusionRequirements, dtUpdateTransfusionRequirements, Common.UpdateFunction.UnitTests);
            Assert.IsTrue(success);
        }


		private void SetupAllCRs(BOL.Patient patient)
		{
			DataTable dtInsertSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(false);
			DataTable dtUpdateSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(true);
			DataTable dtInsertTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(false);
			DataTable dtUpdateTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(true);

			UnitTestUtil.RunSQL("DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + patient.PatientGuid + "'");
		
			// Irradiated
			BOL.PatientTransfusionRequirement pTR = new BOL.PatientTransfusionRequirement(patient, "Irradiated cellular products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
			Common.ComponentRequirement req = Common.ComponentRequirement.IRRADIATE;
			AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);

			// Leuko
			pTR = new BOL.PatientTransfusionRequirement(patient, "Leuko-reduced cellular products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
			req = Common.ComponentRequirement.LEUKORED;
			AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);

			// Washed RBC
			pTR = new BOL.PatientTransfusionRequirement(patient, "Washed RBC products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
			req = Common.ComponentRequirement.WASHEDRBC;
			AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);

			// Washed PLT
			pTR = new BOL.PatientTransfusionRequirement(patient, "Washed PLT products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
			req = Common.ComponentRequirement.WASHEDPLT;
			AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);

			// Sickle cell
			pTR = new BOL.PatientTransfusionRequirement(patient, "Sickle Cell Negative RBC products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
			req = Common.ComponentRequirement.SCNEG;
			AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);

			// CMV Neg
			pTR = new BOL.PatientTransfusionRequirement(patient, "CMV Negative cellular products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
			req = Common.ComponentRequirement.CMVNEG;
			AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);
		
			bool success = BOL.Patient.SaveSIsAndTRs(dtInsertSpecialInstructions, dtUpdateSpecialInstructions, dtInsertTransfusionRequirements, dtUpdateTransfusionRequirements, Common.UpdateFunction.UnitTests);
			Assert.IsTrue(success);
		}


		private void AddDataRow(DataTable dtCrs, Common.ComponentRequirement req, BOL.PatientTransfusionRequirement pTR, BOL.Patient patient)
		{
			DataRow drTRs = dtCrs.NewRow();
			drTRs[TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid] = System.Guid.NewGuid();
			drTRs[TABLES.PatientSpecialInstruction.PatientGuid] = patient.PatientGuid;
			drTRs[TABLES.PatientTransfusionRequirement.TransfusionRequirementText] = pTR.TransfusionRequirement;
			drTRs[TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode] = Common.Utility.GetTransfusionCategoryCodeFromEnum( pTR.TransfusionRequirementCategoryCode );
			drTRs[TABLES.PatientTransfusionRequirement.CreateDate] = pTR.CreateDate;
			drTRs[TABLES.PatientTransfusionRequirement.CreateUser] = pTR.CreateUser;
			drTRs[TABLES.PatientTransfusionRequirement.CreateDivisionCode] = pTR.CreateDivisionCode;
			drTRs[TABLES.PatientTransfusionRequirement.VistaConversionIndicator] = false;
			//drTRs[TABLES.PatientTransfusionRequirement.ComponentRequirementId] = pTR.ComponentRequirementId;
			drTRs[TABLES.PatientTransfusionRequirement.ComponentRequirementId] = (int) req;
			drTRs[TABLES.PatientTransfusionRequirement.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			drTRs[TABLES.PatientTransfusionRequirement.LastUpdateUser] = UnitTestConfig.User_Technologist;
			drTRs[TABLES.PatientTransfusion.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			dtCrs.Rows.Add(drTRs);
		}


		private void SetupAntigenTr(BOL.Patient patient, Guid bloodUnitGuid, bool antibody)
		{
			DataTable dtInsertSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(false);
			DataTable dtUpdateSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(true);
			DataTable dtInsertTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(false);
			DataTable dtUpdateTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(true);
				
			BOL.PatientTransfusionRequirement trAntigen = null;
			if (antibody)
			{
				// Create new TR (antigen neg for C) 
				trAntigen = new BOL.PatientTransfusionRequirement(patient, "Anti-C", Common.TransfusionRequirementCategoryCode.AntibodyIdentified, UnitTestConfig.TestDivisionCode);
			}
			else
			{
				// Create new TR (antigen neg for C) 
				trAntigen = new BOL.PatientTransfusionRequirement(patient, "C", Common.TransfusionRequirementCategoryCode.AntigenNegative, UnitTestConfig.TestDivisionCode);
			}
			DataRow drAntigen = dtInsertTransfusionRequirements.NewRow();

			Guid antigenGuid = Guid.NewGuid();
			drAntigen[TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid] = antigenGuid;
			drAntigen[TABLES.PatientSpecialInstruction.PatientGuid] = patient.PatientGuid;
			drAntigen[TABLES.PatientTransfusionRequirement.TransfusionRequirementText] = trAntigen.TransfusionRequirement;
			drAntigen[TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode] = Common.Utility.GetTransfusionCategoryCodeFromEnum( trAntigen.TransfusionRequirementCategoryCode );
			drAntigen[TABLES.PatientTransfusionRequirement.CreateDate] = trAntigen.CreateDate;
			drAntigen[TABLES.PatientTransfusionRequirement.CreateUser] = trAntigen.CreateUser;
			drAntigen[TABLES.PatientTransfusionRequirement.CreateDivisionCode] = trAntigen.CreateDivisionCode;
			drAntigen[TABLES.PatientTransfusionRequirement.VistaConversionIndicator] = false;
			if (antibody)
			{
				drAntigen[TABLES.PatientTransfusionRequirement.AntibodyTypeId] = 5;
			}
			else
			{
				drAntigen[TABLES.PatientTransfusionRequirement.AntigenTypeId] = 6;
			}
			drAntigen[TABLES.PatientTransfusionRequirement.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			drAntigen[TABLES.PatientTransfusionRequirement.LastUpdateUser] = UnitTestConfig.User_Technologist;
			drAntigen[TABLES.PatientTransfusion.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			dtInsertTransfusionRequirements.Rows.Add(drAntigen);

			BOL.Patient.SaveSIsAndTRs(dtInsertSpecialInstructions, dtUpdateSpecialInstructions, dtInsertTransfusionRequirements, dtUpdateTransfusionRequirements, Common.UpdateFunction.UnitTests);
		}

		private Guid CreateTestData(Guid patientGuid, 
			Common.ComponentClass componentClass, 
			bool specimenRequired, 
			bool associateWithSpecimen,
			string specimenStatusCode, // A: In BB,  C: Converted,  N: NSR,  R: Required
			string orderUrgencyCode, // A: ASAP.  P: Pre-Op,  R: Routine,  S: Stat
			bool originalOrder, // Indicates whether this is the order created at the beginning of test setup
			// Unit assigned to patient
			bool crossmatchRequired, 
			string crossmatchResultCode, // C: Crossmatched
			bool emergencyIssue, // If set to true, specimen tests will not be savedv
			params Guid[] bloodUnitGuids
			) 
		{
			#region clear patient data

			string sql = "DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + patientGuid + "'";
			UnitTestUtil.RunSQL(sql);

			// Clear any data out of specimen test
			sql = "DELETE FROM SpecimenTest WHERE SpecimenTestGuid IN " + 
				"(SELECT SpecimenTestGuid FROM SpecimenTest " +
				"JOIN PatientSpecimen PS ON PS.PatientSpecimenGuid = SpecimenTest.PatientSpecimenGuid " +
				"WHERE PS.PatientGuid =  '" + patientGuid + "')";
			UnitTestUtil.RunSQL(sql);

			sql = "UPDATE Patient SET BloodTypeCode = NULL, RhFactorCode = NULL WHERE PatientGuid = '" + patientGuid + "'";
			UnitTestUtil.RunSQL(sql);

			#endregion


			#region Create Test Data

			// Patient Treatment
			Guid patientTreatmentTestOneGuid = Guid.NewGuid();
			Guid patientTreatmentTestTwoGuid = Guid.NewGuid();

			// Patient Order
			Guid patientOrderTestOneGuid = Guid.NewGuid();
			Guid patientOrderTestTwoGuid = Guid.NewGuid();

			// Ordered Test
			Guid orderedTestTestOneGuid = Guid.NewGuid();
			Guid orderedTestTestTwoGuid = Guid.NewGuid();
			_orderedTestGuid = orderedTestTestTwoGuid;

			// Patient Specimen
			Guid patientSpecimenTestOneGuid = Guid.NewGuid();
			_specimenOneGuid = patientSpecimenTestOneGuid;
			Guid patientSpecimenTestTwoGuid = Guid.NewGuid();
			_specimenTwoGuid = patientSpecimenTestTwoGuid;
			
			// Specimen Tests: User interpretations
			Guid specimenTestAboTestOneGuid = Guid.NewGuid();
			Guid specimenTestRhTestOneGuid = Guid.NewGuid();
			_specimenTestRhTestOneGuid = specimenTestRhTestOneGuid;
			Guid specimenTestAboTestTwoGuid = Guid.NewGuid();
			Guid specimenTestRhTestTwoGuid = Guid.NewGuid();
			_specimenTestRhTestTwoGuid = specimenTestRhTestTwoGuid;
			Guid patientSpecimenTestABScreenGuid = Guid.NewGuid();

			// Specimen Tests: System interpretations
			Guid specimenTestAntiAGuid = Guid.NewGuid();
			Guid specimenTestAntiBGuid = Guid.NewGuid();
			Guid specimenTestAntiABGuid = Guid.NewGuid();
			Guid specimenTestA1CellsGuid = Guid.NewGuid();
			Guid specimenTestBCellsGuid = Guid.NewGuid();

			Guid specimenTestTwoAntiAGuid = Guid.NewGuid();
			Guid specimenTestTwoAntiBGuid = Guid.NewGuid();
			Guid specimenTestTwoAntiABGuid = Guid.NewGuid();
			Guid specimenTestTwoA1CellsGuid = Guid.NewGuid();
			Guid specimenTestTwoBCellsGuid = Guid.NewGuid();
			
			// Ordered Component
			Guid orderedComponentGuid = Guid.NewGuid();
			_orderedComponentGuid = orderedComponentGuid;

			// Ordered Unit
			ArrayList orderedUnitGuids = new ArrayList();
				 
			// First, we need two new Specimen Uids
			string specimenUid1 = DataCreator.GetNewSpecimenUid();
			string specimenUid2 = DataCreator.GetNewSpecimenUid();

			//
			StringBuilder query = new StringBuilder();
			//
			query.Append("BEGIN TRANSACTION T1 ");
			
			// PatientTreatment
			query.Append("INSERT INTO PATIENTTREATMENT (");
			query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
			query.Append(TABLES.PatientTreatment.InPatientIndicator).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append(1).Append(",");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-5))).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			query.Append("INSERT INTO PATIENTTREATMENT (");
			query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
			query.Append(TABLES.PatientTreatment.InPatientIndicator).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientTreatmentTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append(1).Append(",");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-5))).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// First entry into PatientOrder
			query.Append("INSERT INTO PATIENTORDER (");
			query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
			query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientOrder.OrderingProviderLastName).Append(",");
			query.Append(TABLES.PatientOrder.OrderingProviderFirstName).Append(",");
			query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append("BOGUS").Append("',");
			query.Append("'").Append("DOCTOR").Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-5))).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// Second entry into PatientOrder (is this one needed?)
			query.Append("INSERT INTO PATIENTORDER (");
			query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
			query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientOrder.OrderingProviderLastName).Append(",");
			query.Append(TABLES.PatientOrder.OrderingProviderFirstName).Append(",");
			query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientOrderTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientTreatmentTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append("BOGUS").Append("',");
			query.Append("'").Append("DOCTOR").Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-5))).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
						
			// Specimen #1
			query.Append("INSERT INTO PATIENTSPECIMEN (");
			query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenExpirationDate).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddDays(3))).Append("',");
			query.Append("'A',");
			query.Append("'0',");
			query.Append("'").Append(specimenUid1).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-5))).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// Specimen #2
			query.Append("INSERT INTO PATIENTSPECIMEN (");
			query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenExpirationDate).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddDays(3))).Append("',");
			query.Append("'A',");
			query.Append("'0',");
			query.Append("'").Append(specimenUid2).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-5))).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");


			// OrderedTest #1
			query.Append("INSERT INTO ORDEREDTEST (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid).Append("',");
			query.Append("'1',");
			query.Append("'A',");
			query.Append("'P',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-5))).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// OrderedTest #2
			query.Append("INSERT INTO ORDEREDTEST (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid).Append("',");
			query.Append("'" + (int) Common.OrderableTest.TAS +"',");
			if (emergencyIssue)
			{
				query.Append("'" + Common.Utility.GetOrderStatusCodeFromEnum(Common.OrderStatus.Pending) +"',");
				query.Append("'" + Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.NotStarted) +"',");
			}
			else
			{
				query.Append("'" + Common.Utility.GetOrderStatusCodeFromEnum(Common.OrderStatus.Complete) +"',");
				query.Append("'" + Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.Completed) +"',");
			}
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-5))).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");


			#region tech interps
			
			// Specimen (#1) test: A
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAboTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			//query.Append("18,");
			query.Append("65,");
			query.Append("'A',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-5))).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// Specimen (#1) test: P
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestRhTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			//query.Append("21,");
			query.Append("68,");
			query.Append("'P',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-5))).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			if (!emergencyIssue)
			{
				// Specimen (#2) test: A
				query.Append("INSERT INTO SPECIMENTEST (");
				query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
				query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
				query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
				query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
				query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
				query.Append(TABLES.SpecimenTest.TestDate).Append(",");
				query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
				query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
				query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
				query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
				query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
				query.Append("VALUES (");
				query.Append("'").Append(specimenTestAboTestTwoGuid.ToString()).Append("',");
				query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
				query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
				//query.Append("18,");
				query.Append("65,");
				query.Append("'A',");
				query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-4))).Append("',");
				query.Append("'A',");
				query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
				query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
				query.Append("'UnitTestUser',");
				query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
				query.Append("\n");
			
				// Specimen (#2) test: P
				query.Append("INSERT INTO SPECIMENTEST (");
				query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
				query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
				query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
				query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
				query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
				query.Append(TABLES.SpecimenTest.TestDate).Append(",");
				query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
				query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
				query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
				query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
				query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
				query.Append("VALUES (");
				query.Append("'").Append(specimenTestRhTestTwoGuid.ToString()).Append("',");
				query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
				query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
				//query.Append("21,");
				query.Append("68,");
				query.Append("'P',");
				query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(1))).Append("',");
				query.Append("'A',");
				query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
				query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-4))).Append("',");
				query.Append("'UnitTestUser',");
				query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
				query.Append("\n");
			}

			#endregion


			#region system interps (specimen #1)

			// Specimen (#1) test: (system generated Anti-A TAS)
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAntiAGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append( (int) Common.TestType.AntiA_TAS + ",");
			query.Append("'P',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-5))).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// Specimen (#1) test: (system generated Anti-B TAS)
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAntiBGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append( (int) Common.TestType.AntiB_TAS + ",");
			query.Append("'N',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-5))).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// Specimen (#1) test: (system generated Anti-AB TAS)
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAntiABGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append( (int) Common.TestType.AntiAB_TAS + ",");
			query.Append("'X',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-5))).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// Specimen (#1) test: (system generated A1 cells)
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestA1CellsGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append( (int) Common.TestType.A1Cells + ",");
			query.Append("'N',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-5))).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// Specimen (#1) test: (system generated B cells TAS)
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestBCellsGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append( (int) Common.TestType.BCells + ",");
			query.Append("'P',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-5))).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			#endregion


			#region system interps (specimen #2)

			// Specimen (#2) test: (system generated Anti-A TAS)
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestTwoAntiAGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append( (int) Common.TestType.AntiA_TAS + ",");
			query.Append("'P',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-4))).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// Specimen (#2) test: (system generated Anti-B TAS)
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestTwoAntiBGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append( (int) Common.TestType.AntiB_TAS + ",");
			query.Append("'N',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-4))).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// Specimen (#2) test: (system generated Anti-AB TAS)
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestTwoAntiABGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append( (int) Common.TestType.AntiAB_TAS + ",");
			query.Append("'X',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-4))).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// Specimen (#2) test: (system generated A1 cells)
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestTwoA1CellsGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append( (int) Common.TestType.A1Cells + ",");
			query.Append("'N',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(1))).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-4))).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// Specimen (#2) test: (system generated B cells TAS)
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestTwoBCellsGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append( (int) Common.TestType.BCells + ",");
			query.Append("'P',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-4))).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			#endregion


			// Specimen (#2) test: Antibody screen
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(patientSpecimenTestABScreenGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'92',");
			query.Append("'N',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-4))).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");



			// OrderedComponent
			query.Append("INSERT INTO ORDEREDCOMPONENT (");
			query.Append(TABLES.OrderedComponent.OrderedComponentGuid).Append(",");
			query.Append(TABLES.OrderedComponent.PatientOrderGuid).Append(",");
			query.Append(TABLES.OrderedComponent.CprsOrderNumber).Append(",");
			query.Append(TABLES.OrderedComponent.ComponentClassId).Append(",");
			query.Append(TABLES.OrderedComponent.RequiredUnitQuantity).Append(",");
			query.Append(TABLES.OrderedComponent.RequiredDatetime).Append(",");
			query.Append(TABLES.OrderedComponent.SpecimenRequiredIndicator).Append(",");
			if (associateWithSpecimen) query.Append(TABLES.OrderedComponent.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.OrderedComponent.SpecimenStatusCode).Append(",");
			query.Append(TABLES.OrderedComponent.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedComponent.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedComponent.OrderUrgencyCode).Append(",");
			query.Append(TABLES.OrderedComponent.DivisionCode).Append(",");
			query.Append(TABLES.OrderedComponent.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedComponent.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedComponent.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedComponentGuid.ToString()).Append("',");		// ordered component guid
			query.Append("'").Append(patientOrderTestTwoGuid.ToString()).Append("',");	// patient order guid
            query.Append("'").Append(DataCreator.GetNewCprsOrderNumber()).Append("',");
			query.Append((int) componentClass + ",");
			query.Append(10 + ",");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			if (specimenRequired) query.Append(1 + ",");
			else query.Append(0 + ",");							// specimen required 
			if (associateWithSpecimen) query.Append("'" + patientSpecimenTestTwoGuid + "',");	// patient specimen
			query.Append("'" + specimenStatusCode + "',");		// specimen status code
			query.Append("'A',");								// order status code
			query.Append("'P',");								// task status code
			query.Append("'" + orderUrgencyCode + "',");// order urgency code
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-5))).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			bool firstOU = true;
			foreach (Guid bloodUnitGuid in bloodUnitGuids)
			{
				Guid orderedUnitGuid = Guid.NewGuid();
				orderedUnitGuids.Add(orderedUnitGuid);
				if (firstOU)
				{
					_orderedUnitGuid = orderedUnitGuid;
					firstOU = false;
				}
				// Ordered Unit
				query.Append("INSERT INTO ORDEREDUNIT (");
				query.Append(TABLES.OrderedUnit.OrderedUnitGuid).Append(",");
				query.Append(TABLES.OrderedUnit.OrderedComponentGuid).Append(",");
				query.Append(TABLES.OrderedUnit.BloodUnitGuid).Append(",");
				query.Append(TABLES.OrderedUnit.CrossmatchDate).Append(",");
				query.Append(TABLES.OrderedUnit.CrossmatchRequiredIndicator).Append(",");
				if (crossmatchResultCode != null)
				{
					query.Append(TABLES.OrderedUnit.CrossmatchResultCode).Append(",");
					query.Append(TABLES.OrderedUnit.CrossmatchTechId).Append(",");
				}
				query.Append(TABLES.OrderedUnit.RecordStatusCode).Append(",");
				query.Append(TABLES.OrderedUnit.DivisionCode).Append(",");
				query.Append(TABLES.OrderedUnit.LastUpdateDate).Append(",");
				query.Append(TABLES.OrderedUnit.LastUpdateUser).Append(",");
				query.Append(TABLES.OrderedUnit.LastUpdateFunctionId).Append(")");
				query.Append("VALUES (");
				query.Append("'" + orderedUnitGuid.ToString() + "',");
				query.Append("'" + orderedComponentGuid + "',");
				query.Append("'" + (Guid) bloodUnitGuid + "',");
				query.Append("'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',");
				if (crossmatchRequired)	query.Append("1,");
				else query.Append("0,");
				if (crossmatchResultCode != null)
				{
					query.Append("'" + crossmatchResultCode + "',");
					query.Append("'" + Environment.UserName + "',");
				}
				query.Append("'A',");
				query.Append("'" + Common.LogonUser.LogonUserDivisionCode + "',");
				query.Append("'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-5)) + "',");
				query.Append("'" + Environment.UserName + "',");
				query.Append("'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')");
				query.Append("\n");
			}
			//
			//			//Update unit
			//			if (crossmatchResultCode == "C")
			//			{
			//				query.Append("UPDATE BloodUnitStatus SET UnitStatusCode = 'C' WHERE BloodUnitGuid = '" + bloodUnitGuid + "'");
			//			}
			//			else
			//			{
			//				query.Append("UPDATE BloodUnitStatus SET UnitStatusCode = 'S' WHERE BloodUnitGuid = '" + bloodUnitGuid + "'");
			//			}
			//			query.Append("\n");

			query.Append("COMMIT TRANSACTION T1 ");
			//
			UnitTestUtil.RunSQL(query.ToString());
			query.Remove(0, query.Length);

			#endregion
			


			//_patientUnitIssue = new BOL.PatientUnitIssue(new BOL.Patient(_patientGuid), GetVbecsUser(Common.UserRoleType.Technologist));
			//Assertion.IsTrue( _patientUnitIssue.Units != null && _patientUnitIssue.Units.Count == 1 );


			// Delete test data
			#region Clear Test Data

			query.Append("BEGIN TRANSACTION T2 ");
			for (int i=0; i<orderedUnitGuids.Count; i++)
			{
				query.Append("DELETE FROM OrderedUnit WHERE OrderedUnitGuid = '" + (Guid) orderedUnitGuids[i] + "'");
			}
			query.Append("DELETE FROM ORDEREDCOMPONENT WHERE OrderedComponentGuid ='").Append(orderedComponentGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestAboTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestRhTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestAboTestTwoGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestRhTestTwoGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(patientSpecimenTestABScreenGuid).Append("' ").Append("\n");
	
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestAntiAGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestAntiBGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestAntiABGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestA1CellsGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestBCellsGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestTwoAntiAGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestTwoAntiBGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestTwoAntiABGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestTwoA1CellsGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestTwoBCellsGuid).Append("' ").Append("\n");
			
			query.Append("DELETE FROM ORDEREDTEST WHERE OrderedTestGuid ='").Append(orderedTestTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM ORDEREDTEST WHERE OrderedTestGuid ='").Append(orderedTestTestTwoGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTSPECIMEN WHERE PatientSpecimenGuid ='").Append(patientSpecimenTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTSPECIMEN WHERE PatientSpecimenGuid ='").Append(patientSpecimenTestTwoGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTORDER WHERE PatientOrderGuid ='").Append(patientOrderTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTORDER WHERE PatientOrderGuid ='").Append(patientOrderTestTwoGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTTREATMENT WHERE PatientTreatmentGuid ='").Append(patientTreatmentTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTTREATMENT WHERE PatientTreatmentGuid ='").Append(patientTreatmentTestTwoGuid).Append("' ").Append("\n");
			query.Append("COMMIT TRANSACTION T2 ");

			if (originalOrder)
			{
				_cleanupSQL = query.ToString();
			}
			else
			{
				_cleanupSQLTwo = query.ToString();
			}
			
			#endregion

			return orderedComponentGuid;
		}





		private BOL.VbecsUser GetVbecsUser(Common.UserRoleType userRole)
		{
			DataRow dtRow = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 VbecsUser.VbecsUserId, VU.DivisionCode FROM VbecsUser JOIN VbecsUserDivisionRole VU ON VbecsUser.VbecsUserId = VU.VbecsUserId WHERE VbecsUser.SystemUserIndicator = 0 AND VbecsUser.RecordStatusCode = 'A' AND DivisionCode = '" + Common.LogonUser.LogonUserDivisionCode  + "'");
			string updateSql = "UPDATE VbecsUserDivisionRole SET " + TABLES.VbecsUserDivisionRole.UserRoleId + " = " + (int) userRole + ", RecordStatusCode = 'A', LastUpdateDate = '" + DateTime.Now + "' WHERE " + TABLES.VbecsUserDivisionRole.DivisionCode + " = '" + Common.LogonUser.LogonUserDivisionCode + "' AND " + TABLES.VbecsUserDivisionRole.VbecsUserId + " = '" + (string) dtRow[TABLES.VbecsUserDivisionRole.VbecsUserId] + "'";
			UnitTestUtil.RunSQL(updateSql);
			BOL.VbecsUser vbecsUser =  new BOL.VbecsUser( (string) dtRow[TABLES.VbecsUserDivisionRole.VbecsUserId] );
			vbecsUser.LogonToDivision( Common.LogonUser.LogonUserDivisionCode );
			vbecsUser.CurrentDivisionRole.Division.ServiceTypeIndicator = true;
			return vbecsUser;
		}


		private Guid CreatePatient()
		{
			Guid patientGuid = Guid.NewGuid();
			StringBuilder query = new StringBuilder();
			// Patient
			query.Append("INSERT INTO PATIENT (");
			query.Append(TABLES.Patient.PatientGuid + ",");
			query.Append(TABLES.Patient.PatientSsn + ",");
			query.Append(TABLES.Patient.VistaPatientId + ",");
			query.Append(TABLES.Patient.PatientLastName + ",");
			query.Append(TABLES.Patient.PatientFirstName + ",");
			query.Append(TABLES.Patient.PatientDob + ",");
			query.Append(TABLES.Patient.PatientDobCode + ",");
			query.Append(TABLES.Patient.RecordStatusCode + ",");
			query.Append(TABLES.Patient.LastUpdateDate + ",");
			query.Append(TABLES.Patient.LastUpdateUser + ",");
			query.Append(TABLES.Patient.LastUpdateFunctionId + ")");
			query.Append("VALUES(");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(new Random().Next(11111,99999).ToString() + new Random().Next(1111,9999).ToString()).Append("',");
			query.Append("'").Append(DataCreator.GetVistaPatientId().ToString()).Append("',");
			query.Append("'CreateXMatchBUTData',");
			query.Append("'SprocHelper',");
			query.Append("'").Append(DateTime.Today.AddYears(-50)).Append("',");
			query.Append("'V',");
			query.Append("'").Append(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active).ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserName).Append("',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("')");

			UnitTestUtil.RunSQL(query.ToString());
			return patientGuid;
		}



		private string GetValidUser()
		{
			if( _user == null )
			{
				ArrayList usersList = BOL.VbecsUser.GetActiveDivisionUsers( Common.LogonUser.LogonUserDivisionCode );
				_user = ((VbecsUser)usersList[0]).UserName;
			}

			return _user;
		}

		#endregion

	}
}

#endif
